// File_Mpegts - Info for MPEG Transport Stream files
// Copyright (C) 2006-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#include <MediaInfo/Setup.h>
#if defined(MEDIAINFO_MPEGTS_YES)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Multiple/File_MpegTs.h"
#include "MediaInfo/Multiple/File_MpegPs.h"
#include "ZenLib/BitStream.h"
#include "ZenLib/Utils.h"
#include "wx/file.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//***************************************************************************
// Constructor/Destructor
//***************************************************************************

//---------------------------------------------------------------------------
File_MpegTs::File_MpegTs()
{
    TS_Packet_Count=0;
}

//---------------------------------------------------------------------------
File_MpegTs::~File_MpegTs()
{
    for (size_t Pos=0; Pos<TS_Handle.size(); Pos++)
        delete TS_Handle[Pos];
}

//***************************************************************************
// Base
//***************************************************************************

//---------------------------------------------------------------------------
void File_MpegTs::Read_Buffer_Init()
{
    Buffer_MinimumSize=32768*4;
}

//---------------------------------------------------------------------------
void File_MpegTs::Read_Buffer_Continue()
{
    //Look for first Sync word
    if (File_Offset==0)
    {
        while (Buffer_Offset+188<Buffer_Size && CC1(Buffer+Buffer_Offset)!=0x47)
            Buffer_Offset++;
        if (Buffer_Offset+188*4>=Buffer_Size)
            return;
        if (CC1(Buffer+Buffer_Offset+188)!=0x47 || CC1(Buffer+Buffer_Offset+188*2)!=0x47 || CC1(Buffer+Buffer_Offset+188*3)!=0x47)
        {
            File_Offset=File_Size;
            return;
        }
        File_Offset_FirstPacket=Buffer_Offset;
        Stream_Prepare(Stream_General);
    }

    //Fill buffers with demuxed streams
    while (TS_Packet());

    //Jump to the end of the file
    if (File_Size!=(int64u)-1 && File_Offset>8*1024*1024 && File_Size>File_Offset+Buffer_Size+8*1024*1024)
    {
        Details_Add_Element(0, _T("-------------------------------------------"));
        Details_Add_Element(0, _T("---   MPEG-TS, Jumping to end of file   ---"));
        Details_Add_Element(0, _T("-------------------------------------------"));
        File_GoTo=File_Size-8*1024*1024;
    }
}

//---------------------------------------------------------------------------
void File_MpegTs::Read_Buffer_Finalize()
{
    //Streams
    for (size_t PID=0; PID<TS_Handle.size(); PID++)
        if (TS_Handle[PID])
        {
            TS_Handle[PID]->Open_Buffer_Finalize();
            Merge (*TS_Handle[PID]);
        }

    //Fill General
    if (General.empty())
        Stream_Prepare(Stream_General);
         if (!Video.empty() && Video[0](_T("Codec"))==_T("MPEG-2V"))
        Fill(Stream_General, 0, "Format", "MPEG-2TS");
    else if (!Video.empty() && Video[0](_T("Codec"))==_T("MPEG-4V"))
        Fill(Stream_General, 0, "Format", "MPEG-4TS");
    else
        Fill(Stream_General, 0, "Format", "MPEG-1TS");

    return;
}

//***************************************************************************
// Elements
//***************************************************************************

//---------------------------------------------------------------------------
bool File_MpegTs::TS_Packet()
{
    //Enough data?
    if (Buffer_Offset+188>Buffer_Size)
        return false;

    //Tag present?
    if (Buffer[Buffer_Offset]!=0x47)
    {
        //Sync lost
        while (Buffer_Offset+188<Buffer_Size && CC1(Buffer+Buffer_Offset)!=0x47)
            Buffer_Offset++;
        if (Buffer_Offset+188*4>=Buffer_Size)
            return false;
        if (CC1(Buffer+Buffer_Offset+188)!=0x47 || CC1(Buffer+Buffer_Offset+188*2)!=0x47 || CC1(Buffer+Buffer_Offset+188*3)!=0x47)
            return false;
    }

    //Header
    BitStream BS(Buffer+Buffer_Offset, 188);
    BS.Skip(8);                                     //sync_byte
    BS.Skip(1);                                     //transport_error_indicator
    BS.Skip(1);                                     //payload_unit_start_indicator
    BS.Skip(1);                                     //transport_priority
    size_t PID=BS.Get(13);                          //PID
    BS.Skip(2);                                     //transport_scrambling_control
    size_t adaptation_field_control=BS.Get(2);      //adaptation_field_control
    BS.Skip(4);                                     //continuity_counter

    //Details
    if (Config.Details_Get())
    {
        Details_Add_Element(1, _T("Packet"));
        Details_Add_Element(Ztring::ToZtring(PID, 16).MakeUpperCase());
    }

    if (adaptation_field_control==0x0)
    {
        Buffer_Offset+=188;
        return true; //Reserved
    }
    if (adaptation_field_control==0x2 || adaptation_field_control==0x3)
    {
        size_t Adaptation_Size=BS.Get(8);
        for (size_t Pos=0; Pos<Adaptation_Size; Pos++)
            BS.Skip(8);
    }
    Size=188-BS.Offset_Get();
    Buffer_Offset+=BS.Offset_Get();
    if (adaptation_field_control==0x1 || adaptation_field_control==0x3)
    {
        //PIDs
        if (Buffer_Offset+Size>Buffer_Size) ;
        else if (PID==0x0000) ProgramAssociationTable();
        else if (PID==0x0001) ConditionalAccessTable();
        else if (PID>=0x0002
              && PID<=0x000F) Reserved();
        else if (PID==0x1FFF) Null();
        else
            PES(PID);
    }

    Buffer_Offset+=Size;
    TS_Packet_Count++;
    return true;
}

//***************************************************************************
// Elements
//***************************************************************************

#define NAME(ELEMENT_NAME) \
    if (Config.Details_Get()) \
    { \
        Details_Add_Element(ELEMENT_NAME); \
    } \

//---------------------------------------------------------------------------
void File_MpegTs::ProgramAssociationTable()
{
    NAME("Program Association Table")
}

//---------------------------------------------------------------------------
void File_MpegTs::ConditionalAccessTable()
{
    NAME("Conditional Access Table")
}

//---------------------------------------------------------------------------
void File_MpegTs::Reserved()
{
    NAME("Reserved")
}

//---------------------------------------------------------------------------
void File_MpegTs::Null()
{
    NAME("Null")
}

//---------------------------------------------------------------------------
void File_MpegTs::PES(size_t PID)
{
    NAME("Packet Stream")

    //Integrity
    if (Size>184) //A PES can't be more than 184 bytes (188 TS minus 4 bytes header)
        return;

    //Reserve place
    if (PID>=TS_Handle.size())
        TS_Handle.resize(PID+1);
    if (TS_Handle[PID]==NULL)
        TS_Handle[PID]=new File_MpegPs;

    //{wxFile F; F.Open((Ztring(_T("d:\\direct"))+Ztring::ToZtring(PID, 16)).c_str(), wxFile::write_append); F.Write(Buffer+Buffer_Offset, Size);}

    //Open Mpeg
    if (TS_Handle[PID] && TS_Handle[PID]->File_GoTo==0)
    {
        Open_Buffer_Init(TS_Handle[PID], File_Size, File_Offset+Buffer_Offset);
        TS_Handle[PID]->FromTS=true;
        Open_Buffer_Continue(TS_Handle[PID], Buffer+Buffer_Offset, Size);
    }
}

//---------------------------------------------------------------------------
void File_MpegTs::HowTo(stream_t StreamKind)
{
    switch (StreamKind)
    {
        case (Stream_General) :
            Fill_HowTo("Format", "R");
            break;
        case (Stream_Video) :
            break;
        case (Stream_Audio) :
            break;
        case (Stream_Text) :
            break;
        case (Stream_Chapters) :
            break;
        case (Stream_Image) :
            break;
        case (Stream_Menu) :
            break;
        case (Stream_Max) :
            break;
    }
}

} //NameSpace

#endif //MEDIAINFO_MPEGTS_YES


