// File_Mpeg4 - Info for MPEG-4 files
// Copyright (C) 2005-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Information about MPEG-4 files
// Main part
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#include <MediaInfo/Setup.h>
#if defined(MEDIAINFO_MPEG4_YES)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Multiple/File_Mpeg4.h"
#include <ZenLib/Utils.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//***************************************************************************
// Format
//***************************************************************************

//---------------------------------------------------------------------------
void File_Mpeg4::Read_Buffer_Init()
{
    //Buffer
    Element_Level=0;
    Element_Name[0]=0;
    Element_Next[0]=(int64u)-1; //Unlimited

    //Temp
    TimeScale=0;
}

//---------------------------------------------------------------------------
void File_Mpeg4::Read_Buffer_Continue()
{
    while (Buffer_Parse());
}

//***************************************************************************
// Buffer
//***************************************************************************

//---------------------------------------------------------------------------
// ATOM structure
// Size of Data
// Name
// Data
//
bool File_Mpeg4::Buffer_Parse()
{
    //Parse if needed
    if (File_Offset+Buffer_Offset<Element_Next[Element_Level])
    {
        //Enough data?
        if (Buffer_Offset+8>Buffer_Size)
            return false;

        //Going in a lower level
        Element_Level++;
        Element_HeaderSize=8;

        //Element Name
        Element_Name[Element_Level]=CC4(Buffer+Buffer_Offset+4);

        //Element Size
        Element_Size=BigEndian2int32u(Buffer+Buffer_Offset);
        if (Element_Size<8)
        {
            //Special case: To the end of the file
                 if (Element_Size==0)
            {
                Element_Size=File_Size-(File_Offset+Buffer_Offset);
            }
            //Special case: Big files, size is 64-bit
            else if (Element_Size==1)
            {
                //Enough data?
                if (Buffer_Offset+16>Buffer_Size)
                {
                    Element_Level--;
                    return false;
                }

                //Reading Extended size
                Element_Size=BigEndian2int64u(Buffer+Buffer_Offset+8);
                Element_HeaderSize=16;
            }
            //Invalid
            else
            {
                Element_Size=Element_Next[Element_Level]=(int64u)-1;
            }
        }

        //Element Next
        Element_Next[Element_Level]=File_Offset+Buffer_Offset+Element_Size;

        //Integrity of element
        if (Element_Next[Element_Level]<=Element_Next[Element_Level-1])
        {
            //Parsing
            if (!Element_Parse())
                return false; //Not enough bytes
        }
        else
        {
            //Deleting this bad element
            Element_Name[Element_Level]=0;
            Element_Next[Element_Level]=0;
            Element_Level--;
        }
    }

    //To next element
    Buffer_Offset=Element_Next[Element_Level]-File_Offset;
    Element_Name[Element_Level]=0;
    Element_Next[Element_Level]=0;
    Element_Level--;

    if (Element_Level==0)
        return false; //Finnished
    else
        return true; //Next element
}

//***************************************************************************
// Helpers
//***************************************************************************

//---------------------------------------------------------------------------
//Get language string from 2CC
char* File_Mpeg4::Language_Get(int x)
{
    if (x<0x100)
    {
        //Old Quicktime method, I never saw it, skip it
        Language_Result[0] = '_';
        Language_Result[1] = '_';
        Language_Result[2] = '\0';
    }
    else
    {
        Language_Result[3] = '\0';
        Language_Result[2] = (x>> 0&0x1F)+0x60;
        Language_Result[1] = (x>> 5&0x1F)+0x60;
        Language_Result[0] = (x>>10&0x1F)+0x60;
    }
    return Language_Result;
}

//---------------------------------------------------------------------------
//Get Metadata definition from 4CC
File_Mpeg4::method File_Mpeg4::Metadata_Get(std::string &Parameter, int32u Meta)
{
    switch (Meta)
    {
        case Mpeg4::moov_meta___ART : Parameter="Performer"; return Method_String;
        case Mpeg4::moov_meta___alb : Parameter="Album"; return Method_String;
        case Mpeg4::moov_meta___aut : Parameter="Performer"; return Method_String;
        case Mpeg4::moov_meta___cmt : Parameter="Comment"; return Method_String;
        case Mpeg4::moov_meta___cpy : Parameter="Copyright"; return Method_String;
        case Mpeg4::moov_meta___day : Parameter="Encoded_Date"; return Method_String;
        case Mpeg4::moov_meta___des : Parameter="Title/More"; return Method_String;
        case Mpeg4::moov_meta___dir : Parameter="Director"; return Method_String;
        case Mpeg4::moov_meta___dis : Parameter="TermsOfUse"; return Method_String;
        case Mpeg4::moov_meta___edl : Parameter="Tagged_Date"; return Method_String;
        case Mpeg4::moov_meta___fmt : Parameter="Origin"; return Method_String;
        case Mpeg4::moov_meta___gen : Parameter="Genre"; return Method_String;
        case Mpeg4::moov_meta___hos : Parameter="HostComputer"; return Method_String;
        case Mpeg4::moov_meta___inf : Parameter="Title/More"; return Method_String;
        case Mpeg4::moov_meta___key : Parameter="Keywords"; return Method_String;
        case Mpeg4::moov_meta___mak : Parameter="Make"; return Method_String;
        case Mpeg4::moov_meta___mod : Parameter="Model"; return Method_String;
        case Mpeg4::moov_meta___nam : Parameter="Title"; return Method_String;
        case Mpeg4::moov_meta___prd : Parameter="Producer"; return Method_String;
        case Mpeg4::moov_meta___PRD : Parameter="Product"; return Method_String;
        case Mpeg4::moov_meta___prf : Parameter="Performer"; return Method_String;
        case Mpeg4::moov_meta___req : Parameter="SpecialPlaybackRequirements"; return Method_String;
        case Mpeg4::moov_meta___src : Parameter="DistribtedBy"; return Method_String;
        case Mpeg4::moov_meta___swr : Parameter="Encoded_application"; return Method_String;
        case Mpeg4::moov_meta___too : Parameter="Encoded_application"; return Method_String;
        case Mpeg4::moov_meta___wrn : Parameter="Warning"; return Method_String;
        case Mpeg4::moov_meta___wrt : Parameter="ScreenplayBy"; return Method_String;
        case Mpeg4::moov_meta__cpil : Parameter.clear(); return Method_None;
        case Mpeg4::moov_meta__disk : Parameter="Part"; return Method_Binary;
        case Mpeg4::moov_meta__name : Parameter="Title"; return Method_String;
        case Mpeg4::moov_meta__trkn : Parameter="Track"; return Method_Binary;
        case Mpeg4::moov_meta__tmpo : Parameter.clear(); return Method_None;
        default :
            {
                Parameter.clear();
                Parameter.append(1, (char)((Meta&0xFF000000)>>24));
                Parameter.append(1, (char)((Meta&0x00FF0000)>>16));
                Parameter.append(1, (char)((Meta&0x0000FF00)>> 8));
                Parameter.append(1, (char)((Meta&0x000000FF)>> 0));
                return Method_String;
            }
    }
}

//---------------------------------------------------------------------------
//Get Metadata definition from string
File_Mpeg4::method File_Mpeg4::Metadata_Get(std::string &Parameter, const std::string &Meta)
{
         if (Meta=="com.apple.quicktime.copyright") Parameter="Copyright";
    else if (Meta=="com.apple.quicktime.displayname") Parameter="Title";
    else if (Meta=="iTunNORM") Parameter="";
    else if (Meta=="iTunes_CDDB_IDs") Parameter="";
    else Parameter=Meta;
    return Method_String;
}

//***************************************************************************
// Information
//***************************************************************************

//---------------------------------------------------------------------------
void File_Mpeg4::HowTo(stream_t StreamKind)
{
    switch (StreamKind)
    {
        case (Stream_General) :
            Fill_HowTo("Format", "R");
            Fill_HowTo("PlayTime", "R");
            Fill_HowTo("Album", "R");
            Fill_HowTo("Part/Track_Total", "R");
            Fill_HowTo("Movie", "R");
            Fill_HowTo("Track", "R");
            Fill_HowTo("Track/Position", "R");
            Fill_HowTo("Performer", "R");
            Fill_HowTo("Encoded_Application", "R");
            Fill_HowTo("Encoded_Date", "R");
            Fill_HowTo("Encoded_Library", "R");
            Fill_HowTo("Comment", "R");
            Fill_HowTo("ContentType", "R");
            Fill_HowTo("Cover", "R");
            Fill_HowTo("Cover_Datas", "N");
            break;
        case (Stream_Video) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("BitRate", "R");
            Fill_HowTo("Width", "R");
            Fill_HowTo("Height", "R");
            Fill_HowTo("AspectRatio", "N");
            Fill_HowTo("FrameRate", "R");
            break;
        case (Stream_Audio) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("BitRate", "R");
            Fill_HowTo("Channel(s)", "N");
            Fill_HowTo("SamplingRate", "R");
            Fill_HowTo("Resolution", "N");
            break;
        case (Stream_Text) :
            break;
        case (Stream_Chapters) :
            break;
        case (Stream_Image) :
            break;
        case (Stream_Menu) :
            break;
        case (Stream_Max) :
            break;
    }
}

//***************************************************************************
//
//***************************************************************************

} //NameSpace

#endif //MEDIAINFO_MPEG4_YES

