// File_Mpegv - Info for MPEG Video files
// Copyright (C) 2004-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#include <MediaInfo/Setup.h>
#if defined(MEDIAINFO_MPEGV_YES)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Video/File_Mpegv.h"
#include <ZenLib/BitStream.h>
#include <ZenLib/Utils.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
namespace MediaInfoLib
{
//---------------------------------------------------------------------------

//***************************************************************************
// Constants
//***************************************************************************

//---------------------------------------------------------------------------
const float Mpegv_FrameRate[]=
{
    0,
    23.976,
    24,
    25,
    29.97,
    30,
    50,
    59.94,
    60,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
};

//---------------------------------------------------------------------------
const float Mpegv_Ratio1[]=
{
    0,
    1,
    0.6735,
    0.7031, //16/9 PAL
    0.7615,
    0.8055,
    0.8437, //16/9 NTSC
    0.8935,
    0.9157, //4/3 PAL
    0.9815,
    1.0255,
    1.0695,
    1.0950, //4/3 NTSC
    1.1575,
    1.2015,
    0,
};

//---------------------------------------------------------------------------
const float Mpegv_Ratio2[]=
{
    0,
    1,
    (float)4/3,
    (float)16/9,
    2.21,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
};

//---------------------------------------------------------------------------
const char* Mpegv_Chroma[]=
{
    "",
    "4:2:0",
    "4:2:2",
    "4:4:4",
};

//---------------------------------------------------------------------------
const char* Mpegv_Standard[]=
{
    "Composite",
    "PAL",
    "NTSC",
    "SECAM",
    "MAC",
    "",
    "",
    "",
};

//---------------------------------------------------------------------------
const char* Mpegv_Interlacement[]=
{
    "Interlaced",
    "TFF",
    "BFF",
    "PPF",
};

//---------------------------------------------------------------------------
const char* Mpegv_Profile[]=
{
    "",
    "High",
    "Spatial",
    "SNR",
    "Main",
    "Simple",
    "",
    "",
};

//---------------------------------------------------------------------------
const char* Mpegv_Level[]=
{
    "",
    "",
    "",
    "",
    "High",
    "",
    "High-1440",
    "",
    "Main",
    "",
    "Low",
    "",
    "",
    "",
    "",
    "",
};

//---------------------------------------------------------------------------
const char* Mpegv_FrameType[]=
{
    "",
    "I",
    "P",
    "B",
    "D",
    "",
    "",
    "",
};

//***************************************************************************
// Format
//***************************************************************************

//---------------------------------------------------------------------------
void File_Mpegv::Read_Buffer_Init()
{
    //Elements
    Buffer_Next=0;
    Element_Size=0;

    //Temp
    Time_Begin_Seconds=Error;
    Time_Begin_Frames=Error;
    Time_End_Seconds=Error;
    Time_End_Frames=Error;
    Time_End_NeedComplete=false;
    Width=0;
    Height=0;
    RatioValue=0;
    FrameRate=0;
    BitRate=0;
    Errors=0;
    User_Start_Count=0;
}

//---------------------------------------------------------------------------
void File_Mpegv::Read_Buffer_Continue()
{
    //Coherancy
    if (File_Offset==0)
    {
        //-Element_Size
        if (Buffer_Size<8)
        {
            File_Offset=File_Size;
            return;
        }
        //-Detect DAT files, and the parser is not enough precise to detect them later
        if (CC4(Buffer)==CC4("RIFF"))
        {
            File_Offset=File_Size;
            return;
        }
        //-Detect TS files, and the parser is not enough precise to detect them later
        if (Buffer_Size>=188*4)
        {
            //Look for first Sync word
            while (Buffer_Offset<188 && CC1(Buffer+Buffer_Offset)!=0x47)
                Buffer_Offset++;
            if (Buffer_Offset<188 && CC1(Buffer+Buffer_Offset+188)==0x47 && CC1(Buffer+Buffer_Offset+188*2)==0x47 && CC1(Buffer+Buffer_Offset+188*3)==0x47)
            {
                File_Offset=File_Size;
                return;
            }
            Buffer_Offset=0;
        }
    }

    //Look for first Sync word
    Buffer_Offset=0;
    while (Buffer_Offset+3<Buffer_Size && CC3(Buffer+Buffer_Offset)!=0x000001)
        Buffer_Offset++;
    if (Buffer_Offset+3>=Buffer_Size)
    {
        File_Offset=File_Size;
        return;
    }

    //Parse
    while (Buffer_Parse());
}

//---------------------------------------------------------------------------
void File_Mpegv::Read_Buffer_Finalize()
{
    //Tags with multiple location
    //-AspectRatio
    float AspectRatio=0;
    if (MPEG_Version==2)
    {
        if (RatioValue==1 && Height!=0)
            AspectRatio=Width/Height;
        else
            AspectRatio=Mpegv_Ratio2[RatioValue];
    }
    else
    {
        if (Height!=0)
            AspectRatio=(float)Width/Height/Mpegv_Ratio1[RatioValue];;
    }

    //-Version
    if (MPEG_Version==2)
    {
        Fill(Stream_General, 0, "Format", "MPEG-2V");
        Fill("Codec", "MPEG-2V");
        Fill("Codec/String", "MPEG-2 Video");
    }
    else
    {
        Fill(Stream_General, 0, "Format", "MPEG-1V");
        Fill("Codec", "MPEG-1V");
        Fill("Codec/String", "MPEG-1 Video");
        Fill("Interlacement", "PPF");
    }
    Fill("Width", Width);
    Fill("Height", Height);
    Fill("AspectRatio", AspectRatio);
    Fill("FrameRate", FrameRate);

    if (BitRate==0x3FFFF)
        Fill("BitRate_Mode", "VBR");
    else
    {
        Fill("BitRate_Mode", "CBR");
        Fill("BitRate", BitRate*400);
    }

    if (Library.size()>=8)
    {
        Fill("Encoded_Library", Library);
        Fill(Stream_General, 0, "Encoded_Library", Library);
    }

    //Calculate InitTime with Framerate
    if (Time_End_NeedComplete && Config.ParseSpeed_Get()!=1)
        Time_End_Seconds=Error;
    if (Time_End_Seconds!=Error)
    {
        size_t Time_Begin=Time_Begin_Seconds*1000;
        size_t Time_End =Time_End_Seconds*1000;
        Time_End_Frames++; //+1 to count the last frame
        if (FrameRate)
        {
            Time_Begin+=(size_t)(Time_Begin_Frames*1000/FrameRate);
            Time_End  +=(size_t)(Time_End_Frames  *1000/FrameRate);
        }
        if (!Video.empty()) //To avoid too strange behaviours
            Video[0](_T("PlayTime")).From_Number(Time_End-Time_Begin);
    }
}

//***************************************************************************
// Buffer
//***************************************************************************

//---------------------------------------------------------------------------
bool File_Mpegv::Buffer_Parse()
{
    //Enough data?
    if (Buffer_Offset+4>Buffer_Size)
        return false;

    //Element size
    Buffer_Next=Buffer_Offset+4;
    while(Buffer_Next+3<Buffer_Size && CC3(Buffer+Buffer_Next)!=0x000001)
        Buffer_Next++;
    if (Buffer_Next+3>=Buffer_Size)
    {
        //No next element
        if (File_Offset+Buffer_Next+3>=File_Size)
            Buffer_Next=Buffer_Size; //This is the last frame
        else if (Buffer_Next-Buffer_Offset>=32*1024) //False positive, no element should have more than this size
        {
            File_Offset=File_Size;
            return false;
        }
        else
            return false; //Other byte will come
    }
    Element_Size=Buffer_Next-Buffer_Offset-4;

    //Element name
    Element_Name=BigEndian2int8u(Buffer+Buffer_Offset+3);

    //Parse
    Element_Parse();

    //No need of more
    if (File_GoTo)
        return false;

    //Go after the last frame
    Buffer_Offset+=Element_Size;
    while (Buffer_Offset<Buffer_Size && Buffer[Buffer_Offset]==0x00)
        Buffer_Offset++;
    if (Buffer_Offset>=Buffer_Size)
        return false;
    else if (CC3(Buffer+Buffer_Offset-2)!=0x000001)
        return true; // No right bytes found
    Buffer_Offset-=2;
    return true;
}

//---------------------------------------------------------------------------
// Element parse
//
bool File_Mpegv::Element_Parse()
{
    //Details
    if (Config.Details_Get()/* &&
       (Element_Name<0x01
     || Element_Name>0xAF)*/)
    {
        Ztring Pos1; Pos1.From_Number(Element_Name, 16);
        Ztring Pos2;
        Pos2.resize(2-Pos1.size(), _T('0'));
        Pos2+=Pos1;
        Pos2.MakeUpperCase();
        Details_Add_Element(1, Pos2);
    }

    //Header
    Buffer_Offset+=4;

    //Parse
    switch (Element_Name)
    {
        case 0x00: Picture_Start(); break;
        case 0xB2: User_Start(); break;
        case 0xB3: Sequence_Header(); break;
        case 0xB4: Sequence_Error(); break;
        case 0xB5: Extension_Start(); break;
        case 0xB7: Sequence_End(); break;
        case 0xB8: Group_Start(); break;
        case 0xB9: System_Start(); break;
        default:
            if (Element_Name>=0x01
             && Element_Name<=0xAF) Slice_Start();
            else
            {
                //This is maybe not MPEG Video
                Clear();
                File_Offset=File_Size;
            }
    }

    return true;
}

//***************************************************************************
// Elements
//***************************************************************************

#define NAME(ELEMENT_NAME) \
    if (Config.Details_Get()) \
    { \
        Details_Add_Element(ELEMENT_NAME); \
    } \

#define INTEGRITY_GENERAL() \
    if (Count_Get(Stream_General)==0) \
    { \
        if (Config.Details_Get()) \
        { \
            /*Details_Add_Info(Error, "Should not be here", Count_Get(Stream_General));*/ \
        } \
        Errors++; \
        if (Errors>10000) \
            File_Offset=File_Size; \
        return; \
    } \

#define INTEGRITY(ELEMENT_SIZE) \
    if (Element_Size!=ELEMENT_SIZE) \
    { \
        if (Config.Details_Get()) \
        { \
            Details_Add_Info(Error, "Integrity error", ELEMENT_SIZE); \
        } \
       return; \
    } \

//---------------------------------------------------------------------------
// Packet "00", Picture Start
//
void File_Mpegv::Picture_Start()
{
    NAME("Picture_Start")

    //Integrity
    if (Time_End_Seconds!=Error)
        Time_End_Frames++;

    //Reading
    BitStream BS(Buffer+Buffer_Offset, Element_Size);
    BS.Skip(10);                                    //temporal_reference
    size_t picture_coding_type=BS.Get(3);           //picture_coding_type

    //Details
    if (Config.Details_Get())
    {
        Details_Add_Info(Error, "FrameType", Mpegv_FrameType[picture_coding_type]);
        if (Time_End_Seconds!=Error)
        {
            size_t Time_Begin=Time_Begin_Seconds*1000;
            size_t Time_End =Time_End_Seconds*1000;
            if (FrameRate)
            {
                Time_Begin+=(size_t)(Time_Begin_Frames*1000/FrameRate);
                Time_End  +=(size_t)(Time_End_Frames  *1000/FrameRate);
            }
            Details_Add_Info(Error, "Time", Time_End-Time_Begin);
        }
    }
}

//---------------------------------------------------------------------------
// Packet "01" --> "AF", Slice Start
//
void File_Mpegv::Slice_Start()
{
    NAME("Slice_Start")
    INTEGRITY_GENERAL()

    //Detection is finnished
    if (Count_Get(Stream_Video)>0 && File_Size!=(int64u)-1 && Config.ParseSpeed_Get()<=0.01 && File_Size>File_Offset+Buffer_Size+4*1024*1024)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Element(1, _T("------------------------------------------"));
            Details_Add_Element(1, _T("---   MPEG-V, Jumping to end of file   ---"));
            Details_Add_Element(1, _T("------------------------------------------"));
        }

        //Jumping
        File_GoTo=File_Size-4*1024*1024;
        Time_End_Seconds=Error;
        Time_End_Frames=Error;
    }
}

//---------------------------------------------------------------------------
// Packet "B2", User Start
//
void File_Mpegv::User_Start()
{
    NAME("User_Start")

    //In case of User start is not used for library name
    if (User_Start_Count>2)
        return;
    User_Start_Count++;

    //Reading
    size_t Library_Offset=0;
    size_t Library_Size=Element_Size;

    //-Reject junk after the name
    while (Library_Size>0 && (Buffer[Buffer_Offset+Library_Offset+Library_Size-1]<0x20 || Buffer[Buffer_Offset+Library_Offset+Library_Size-1]>0x7D))
        Library_Size--;
    if (Library_Size==0)
        return;

    //-Reject junk before the name
    size_t Library_Offset_Final=Library_Offset+Library_Size-1;
    while (Buffer_Offset+Library_Offset_Final>=Buffer_Offset+Library_Offset && Buffer[Buffer_Offset+Library_Offset_Final]>=0x20 && Buffer[Buffer_Offset+Library_Offset_Final]<=0x7D)
        Library_Offset_Final--;
    if (Buffer_Offset+Library_Offset_Final>=Buffer_Offset+Library_Offset)
    {
        Library_Size -=Library_Offset_Final-Library_Offset;
        Library_Offset=Library_Offset_Final;
    }

    //Filling
    Library.From_Local((const char*)Buffer+Buffer_Offset+Library_Offset, Library_Size);

    //Details
    if (Config.Details_Get())
    {
        Details_Add_Info(Library_Offset, "Datas", Library);
    }
}

//---------------------------------------------------------------------------
// Packet "B3", Sequence Header
//
void File_Mpegv::Sequence_Header()
{
    NAME("Sequence_Header")
    if (!General.empty())
        return;

    BitStream BS(Buffer+Buffer_Offset, Element_Size);
    Width=BS.Get(12);                               //horizontal_size_value
    Height=BS.Get(12);                              //vertical_size_value
    RatioValue=BS.Get(4);                           //aspect_ratio_information
    int32u FrameRate_Code=BS.Get(4);                //frame_rate_code
    BitRate=BS.Get(18);                             //bit_rate_value
    BS.Skip(1);                                     //marker_bit
    BS.Skip(10);                                    //vbv_buffer_size_value
    BS.Skip(1);                                     //constrained_parameters_flag
    int32u Intra   =BS.Get(1);                      //load_intra_quantiser_matrix
    if (Intra)
    {                                               //intra_quantiser_matrix[64]
        for (size_t Pos=0; Pos<8; Pos++) {BS.Skip(32); BS.Skip(32);}
    }
    int32u Intra_No=BS.Get(1);                      //load_non_intra_quantiser_matrix
    if (Intra_No)
    {                                               //non_intra_quantiser_matrix[64]
        for (size_t Pos=0; Pos<8; Pos++) {BS.Skip(32); BS.Skip(32);}
    }

    //Integrity
    BS.Byte_Align();
    INTEGRITY(BS.Offset_Get())

    //Calculations
    FrameRate=Mpegv_FrameRate[FrameRate_Code];

    //Details
    if (Config.Details_Get())
    {
        Details_Add_Info(Error, "Width", Width);
        Details_Add_Info(Error, "Height", Height);
        Details_Add_Info(Error, "AspectRatio Code", RatioValue);
        Details_Add_Info(Error, "FrameRate", FrameRate);
        Details_Add_Info(Error, "BitRate Code", BitRate);
        Details_Add_Info(Error, "Matrix_Intra", Intra);
        Details_Add_Info(Error, "Matrix_Intra_No", Intra_No);
    }

    //Filling
    Stream_Prepare(Stream_General);
    Stream_Prepare(Stream_Video);
    if (Intra || Intra_No)
    {
        Fill("Codec_Settings", "CustomMatrix");
        Fill("Codec_Settings/Matrix", "Custom");
    }
    else
        Fill("Codec_Settings/Matrix", "Standard");
}

//---------------------------------------------------------------------------
// Packet "B4", Sequence Error
//
void File_Mpegv::Sequence_Error()
{
    NAME("Sequence_Error")
    INTEGRITY_GENERAL()
}

//---------------------------------------------------------------------------
// Packet "B5", Extension Start
//
void File_Mpegv::Extension_Start()
{
    NAME("Extension_Start")
    INTEGRITY_GENERAL()

    //Reading
    BitStream BS(Buffer+Buffer_Offset, Element_Size);
    size_t ID=BS.Get(4);                            //extension_start_code_identifier
    MPEG_Version=2; //Extension_Start only exists in MPEG-2 specs

    //sequence_extension
         if (ID==1)
    {
        NAME("Sequence Extension")
        BS.Skip(1);                                 //profile_and_level_indication, escape
        int32u Profile=BS.Get(3);                   //profile_and_level_indication, profile
        int32u Level=BS.Get(4);                     //profile_and_level_indication, level
        int32u Progressive=BS.Get(1);               //progressive_sequence
        int32u Chroma=BS.Get(2);                    //chroma_format
        int32u Width_Ext=BS.Get(2);                 //horizontal_size_extension
        int32u Height_Ext=BS.Get(2);                //vertical_size_extension
        int32u BitRate_Ext=BS.Get(12);              //bit_rate_extension
        BS.Skip(1);                                 //marker_bit
        BS.Skip(8);                                 //vbv_buffer_size_extension
        BS.Skip(1);                                 //low_delay
        int32u FrameRate_N=BS.Get(2);               //frame_rate_extension_n
        int32u FrameRate_D=BS.Get(5);               //frame_rate_extension_d

        //Integrity
        BS.Byte_Align();
        INTEGRITY(BS.Offset_Get())

        //Details
        if (Config.Details_Get())
        {
            Details_Add_Info(Error, "Profile", Profile);
            Details_Add_Info(Error, "Level", Level);
            Details_Add_Info(Error, "Progressive", Progressive);
            Details_Add_Info(Error, "Chroma", Chroma);
            Details_Add_Info(Error, "Width_Ext", Width_Ext);
            Details_Add_Info(Error, "Height_Ext", Height_Ext);
            Details_Add_Info(Error, "BitRate_Ext", BitRate_Ext);
        }

        //Filling
        Width+=0x1000*Width_Ext;
        Height+=0x1000*Height_Ext;
        BitRate+=0x4000*BitRate_Ext;
        if (FrameRate_D)
            FrameRate=FrameRate_N/FrameRate_D;
        if (Progressive && Get(Stream_Video, 0, _T("Interlacement")).empty())
            Fill("Interlacement", Mpegv_Interlacement[3]);
        if (Get(Stream_Video, 0, _T("Chroma")).empty())
            Fill("Chroma", Mpegv_Chroma[Chroma]);
        if (Profile<8 && Level<16 && Get(Stream_Video, 0, _T("Codec_Profile")).empty())
            Fill("Codec_Profile", Ztring().From_Local(Mpegv_Profile[Profile])+_T("@")+Ztring().From_Local(Mpegv_Level[Level]));

    }

    //sequence_display_extension
    else if (ID==2)
    {
        NAME("Sequence Display Extension")
        //Reading
        int32u Standard=BS.Get(3);                  //video_format
        if (BS.Get(1))                              //colour_description
        {
            BS.Skip(8);                             //colour_primaries
            BS.Skip(8);                             //transfer_characteristics
            BS.Skip(8);                             //matrix_coefficients
        }
        BS.Skip(14);                                //display_horizontal_size
        BS.Skip(1);                                 //marker_bit
        BS.Skip(14);                                //display_vertical_size

        //Integrity
        BS.Byte_Align();
        INTEGRITY(BS.Offset_Get())

        //Details
        if (Config.Details_Get())
        {
            Details_Add_Info(Error, "Standard", Standard);
        }

        //Filling
        if (Standard<8 && Get(Stream_Video, 0, _T("Standard")).empty())
            Fill("Standard", Mpegv_Standard[Standard]);
    }
    //
    else if (ID==3)
    {
        NAME("Quant Matrix Extension")
    }
    //
    else if (ID==4)
    {
        NAME("Copyright")
    }
    //
    else if (ID==5)
    {
        NAME("Sequence Scalable")
    }
    //
    else if (ID==7)
    {
        NAME("Picture Display")
    }
    //
    else if (ID==8)
    {
        NAME("Picture Coding")
        //Reading
        BS.Skip(4);                                 //f_code[0][0] (forward horizontal)
        BS.Skip(4);                                 //f_code[0][1] (forward vertical)
        BS.Skip(4);                                 //f_code[1][0] (backward horizontal)
        BS.Skip(4);                                 //f_code[1][1] (backward vertical)
        BS.Skip(2);                                 //intra_dc_precision
        int32u PictureStructure=BS.Get(2);          //picture_structure
        int32u TopField=BS.Get(1);                  //top_field_first
        BS.Skip(1);                                 //frame_pred_frame_dct
        BS.Skip(1);                                 //concealment_motion_vectors
        BS.Skip(1);                                 //q_scale_type
        BS.Skip(1);                                 //intra_vlc_format
        BS.Skip(1);                                 //alternate_scan
        BS.Skip(1);                                 //repeat_first_field
        BS.Skip(1);                                 //chroma_420_type
        BS.Skip(1);                                 //progressive_frame
        if (BS.Get(1))                              //composite_display_flag
        {
            BS.Skip(1);                             //v_axis
            BS.Skip(3);                             //field_sequence
            BS.Skip(1);                             //sub_carrier
            BS.Skip(7);                             //burst_amplitude
            BS.Skip(8);                             //sub_carrier_phase
        }

        //Integrity
        BS.Byte_Align();
        INTEGRITY(BS.Offset_Get())

        //Details
        if (Config.Details_Get())
        {
            Details_Add_Info(Error, "Picture Structure", PictureStructure);
        }

        //Filling
        size_t Interlacement;
        if (PictureStructure==3)
        {
            if (TopField)
                Interlacement=1; //TopField
            else
                Interlacement=2; //BottomField
        }
        else
            Interlacement=PictureStructure; //Depend of 1st field found
        if (Get(Stream_Video, 0, _T("Interlacement")).empty())
            Fill("Interlacement", Mpegv_Interlacement[Interlacement]);
    }
    //
    else if (ID==9)
    {
        NAME("Picture Spatial Scalable")
    }
    //
    else if (ID==10)
    {
        NAME("Picture Temporal Scalable")
    }
    //
    else
    {
        NAME("Reserved")
    }
}

//---------------------------------------------------------------------------
// Packet "B7", Sequence_End
//
void File_Mpegv::Sequence_End()
{
    NAME("Sequence_End")
    INTEGRITY_GENERAL()
}

//---------------------------------------------------------------------------
// Packet "B8", Group Start
//
void File_Mpegv::Group_Start()
{
    NAME("Group_Start")
    INTEGRITY_GENERAL()

    //Integrity
    if (Element_Size>5 && Buffer[Buffer_Offset+4]!=0x00 && Buffer[Buffer_Offset+5]!=00)
        return;

    //Reading
    BitStream BS(Buffer+Buffer_Offset, Element_Size);
    BS.Skip(1);                                     //time_code, drop_frame_flag
    size_t Hours  =BS.Get(5);                       //time_code, time_code_hours
    size_t Minutes=BS.Get(6);                       //time_code, time_code_minutes
    BS.Skip(1);                                     //time_code, marker_bit
    size_t Seconds=BS.Get(6);                       //time_code, time_code_seconds
    size_t Frames =BS.Get(6);                       //time_code, time_code_pictures
    BS.Skip(1);                                     //closed_gop
    BS.Skip(1);                                     //broken_link

    //Integrity
    BS.Byte_Align();
    INTEGRITY(BS.Offset_Get())

    //Calculating
    if (Time_Begin_Seconds==Error)
    {
        //Save begin time before
        Time_Begin_Seconds=60*60*Hours+60*Minutes+Seconds;
        Time_Begin_Frames =Frames;
    }
    Time_End_Seconds=60*60*Hours+60*Minutes+Seconds;
    Time_End_Frames =Frames-1; //We must not count the next picture
    if (Time_End_Seconds+Time_End_Frames==0)
        Time_End_NeedComplete=true; //Will always be empty...

    //Details
    if (Config.Details_Get())
    {
        Details_Add_Info(1, "Time Code, Hours", Hours);
        Details_Add_Info(6, "Time Code, Minutes", Minutes);
        Details_Add_Info(13, "Time Code, Seconds", Seconds);
        Details_Add_Info(19, "Time Code, Frames", Frames);
    }
}

//---------------------------------------------------------------------------
// Packet "B9" --> "FF" , System_Start
//
void File_Mpegv::System_Start()
{
    NAME("System_Start")
    INTEGRITY_GENERAL()
}

//---------------------------------------------------------------------------
void File_Mpegv::HowTo(stream_t StreamKind)
{
    switch (StreamKind)
    {
        case (Stream_General) :
            Fill_HowTo("Format", "R");
            Fill_HowTo("Encoded_Application", "R");
            break;
        case (Stream_Video) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("BitRate", "R");
            Fill_HowTo("Width", "R");
            Fill_HowTo("Height", "R");
            Fill_HowTo("AspectRatio", "R");
            Fill_HowTo("FrameRate", "R");
            break;
        case (Stream_Audio) :
            break;
        case (Stream_Text) :
            break;
        case (Stream_Chapters) :
            break;
        case (Stream_Image) :
            break;
        case (Stream_Menu) :
            break;
        case (Stream_Max) :
            break;
    }
}

} //NameSpace

#endif //MEDIAINFO_MPEGV_YES

