// File_MpegPs - Info for MPEG files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#include <MediaInfo/Setup.h>
#if defined(MEDIAINFO_MPEGPS_YES)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Multiple/File_MpegPs.h"
#if defined(MEDIAINFO_MPEGV_YES)
    #include "MediaInfo/Video/File_Mpegv.h"
#endif
#if defined(MEDIAINFO_MPEG4V_YES)
    #include "MediaInfo/Video/File_Mpeg4v.h"
#endif
#if defined(MEDIAINFO_MPEGA_YES)
    #include "MediaInfo/Audio/File_Mpega.h"
#endif
#if defined(MEDIAINFO_AC3_YES)
    #include "MediaInfo/Audio/File_Ac3.h"
#endif
#if defined(MEDIAINFO_DTS_YES)
    #include "MediaInfo/Audio/File_Dts.h"
#endif
#include <ZenLib/Utils.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// MPEG Program Stream 1&2 - Codes
// Header code = 0x000001
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// End                              0xB9
// Start                            0xBA
// System                           0xBB
// Program map                      0xBC
// Private                          0xBD        (Used for AC3/DTS/PCM in DVD)
// Padding                          0xBE
// Private                          0xBF        (Navigation data)
// MPEG Audio                       0xC0 - 0xDF
// MPEG Video                       0xE0 - 0xEF
// ECM?                             0xF0
// EMM?                             0xF1
// DSMCC?                           0xF2
// ISO 13522?                       0xF3
// ITUT Type A to E?                0xF4 - 0xF8
// PS Directory                     0xFF
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// MPEG Program Stream 1&2 - PES
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Start                            3 bytes - 0x000001
// Code                             1 byte
// Element_Size                     2 bytes (Not always)
// Datas                            x
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//***************************************************************************
// Constructor/Destructor
//***************************************************************************

//***************************************************************************
// Format
//***************************************************************************

//---------------------------------------------------------------------------
void File_MpegPs::Read_Buffer_Init()
{
    MPEG_Version=0;
    Video_Unlimited=false;
    Element_Size=0;
    Element_HeaderSize=0;
    Element_Next=0;
    Buffer_MinimumSize=100;
    FromTS=false;
}

//---------------------------------------------------------------------------
void File_MpegPs::Read_Buffer_Continue()
{
    //Integrity
    if (File_Offset==0 && Detect_NonMPEGPS())
        return;

    //Detect MPEG Version
    if (MPEG_Version==0)
        Detect_MPEG_Version();

    if (!Video_Unlimited)
    {
        //Look for next frame
        if (NextFrame()==Error)
            return;
    }
    else
    {
        //Parse the old unlimited datas before continuing
        Element_Next=0;
        PES_Packet_ElementSizes_Unlimited();
        Element_Parse();
    }

    //Parse with PES parser
    while (PES_Packet());

    //Jumping to the end of the file
    Detect_EOF_Needed();
}

//---------------------------------------------------------------------------
void File_MpegPs::Read_Buffer_Finalize()
{
    //Video
    for (size_t Stream_Pos=0; Stream_Pos<Video_Handle.size(); Stream_Pos++)
        if (Video_Handle[Stream_Pos])
        {

            Open_Buffer_Finalize(Video_Handle[Stream_Pos]);
            if (Merge(*Video_Handle[Stream_Pos])>0)
            {
                if (Video_TimeStamp[Stream_Pos].Have_PTS)
                    Fill("Delay", Video_TimeStamp[Stream_Pos].PTS/90);

                //TimeStamp
                if (Get(Stream_Video, 0, _T("PlayTime")).empty())
                    Fill(Stream_Video, 0, "PlayTime", (Video_TimeStamp_End[Stream_Pos].PTS-Video_TimeStamp[Stream_Pos].PTS)/90);

                //Tags in MPEG Video
                Fill(Stream_General, 0, "Encoded_Library", Get(Stream_Video, StreamPos_Last, _T("Encoded_Library")));
            }

            delete Video_Handle[Stream_Pos];
            Video_Handle[Stream_Pos]=NULL;
        }

    //Audio
    for (size_t Stream_Pos=0; Stream_Pos<Audio_Handle.size(); Stream_Pos++)
        if (Audio_Handle[Stream_Pos])
        {

            Open_Buffer_Finalize(Audio_Handle[Stream_Pos]);
            if (Merge(*Audio_Handle[Stream_Pos])>0)
            {
                if (Audio_TimeStamp[Stream_Pos].Have_PTS)
                    Fill("Delay", Audio_TimeStamp[Stream_Pos].PTS/90);
            }

            delete Audio_Handle[Stream_Pos];
            Audio_Handle[Stream_Pos]=NULL;
        }

    //Private1
    for (size_t Stream_Pos=0; Stream_Pos<Private1_Handle.size(); Stream_Pos++)
        if (Private1_Handle[Stream_Pos])
        {

            Open_Buffer_Finalize(Private1_Handle[Stream_Pos]);
            if (Merge(*Private1_Handle[Stream_Pos])>0)
            {
                if (Private1_TimeStamp[Stream_Pos].Have_PTS)
                    Fill("Delay", Private1_TimeStamp[Stream_Pos].PTS/90);
            }

            delete Private1_Handle[Stream_Pos];
            Private1_Handle[Stream_Pos]=NULL;
        }

    //Fill General
    if (General.empty())
        Stream_Prepare(Stream_General);

         if (!Video.empty() && Video[0](_T("Codec"))==_T("MPEG-2V"))
        Fill(Stream_General, 0, "Format", "MPEG-2PS");
    else if (!Video.empty() && Video[0](_T("Codec"))==_T("MPEG-4V"))
        Fill(Stream_General, 0, "Format", "MPEG-4PS");
    else if (MPEG_Version==2)
        Fill(Stream_General, 0, "Format", "MPEG-2PS");
    else if (MPEG_Version==1)
        Fill(Stream_General, 0, "Format", "MPEG-1PS");

    /*
    //Video BitRate
    if (!From_TS && Video.size()==1 && !General.empty() && General[0](_T("PlayTime")).To_int32s()!=0)
    {
        //Estimited video BitRate
        int32s OveralBitRate_Estimated=File_Size*8*1000/General[0](_T("PlayTime")).To_int32s();
        for (size_t I1=0; I1<Audio.size(); I1++)
            OveralBitRate_Estimated-=Audio[I1](_T("BitRate")).To_int32s();
        int32s OveralBitRate_Theoric=Video[0](_T("BitRate")).To_int32s();
        if (OveralBitRate_Theoric-OveralBitRate_Estimated>100000) // Difference >100 Kbps (overhead estimation)
        {
            //Video BitRate is false!
            Video[0](_T("BitRate")).From_Number((float32_int32s(((float)OveralBitRate_Estimated)/400)-125)*400); // 50 Kbps for Overhead
            Video[0](_T("BitRate_Mode"))=_T("VBR");
        }
    }
    */
}

//***************************************************************************
// Buffer
//***************************************************************************

//---------------------------------------------------------------------------
bool File_MpegPs::PES_Packet()
{
    //Calculating Element name
    if (!PES_Packet_NextFrame())
        return false;
    Element_Name=Buffer[Buffer_Offset+3];

    //Getting sizes
    if (!PES_Packet_ElementSizes_Get())
        return false;

    if (!PES_Packet_ElementName_IsOK())
        return false;

    //Parsing
    Element_Parse();

    return true;
}

//---------------------------------------------------------------------------
//
bool File_MpegPs::Element_Parse()
{
    //Info
    Ztring Pos1; Pos1.From_Number(Element_Name, 16);
    Ztring Pos2;
    Pos2.resize(2-Pos1.size(), _T('0'));
    Pos2+=Pos1;
    Pos2.MakeUpperCase();
    Details_Add_Element(1, Pos2);

    //Positioning
    Buffer_Offset+=Element_HeaderSize;
    Element_Size-=Element_HeaderSize;

    //Streams
    if (0) ;
    else if (Element_Name==0xBA) Start();
    else if (Element_Name==0xBB) System();
    else if (Element_Name==0xBD) Private1();
    else if (Element_Name==0xBE) Padding();
    else if (Element_Name==0xBF) Private2();
    else if (Element_Name>=0xC0
          && Element_Name<=0xDF) Mpeg1a();
    else if (Element_Name>=0xE0
          && Element_Name<=0xEF) Mpeg1v();

    Buffer_Offset+=Element_Size;
    return true;
}

//---------------------------------------------------------------------------
// Packet header
bool File_MpegPs::PES_Packet_Header()
{
    //There is a PES Header with size and long packet header
    if (Buffer_Offset+2>=Buffer_Size)
        return false; //No enough bytes

    //Getting Element size
    int32u PES_Packet_Size=BigEndian2int16u(Buffer+Buffer_Offset+Element_HeaderSize); //PES_packet_length
    Element_Size=6+PES_Packet_Size;
    Element_HeaderSize=6;

    //Parsing Element size
    if (PES_Packet_Size==0)
    {
        //Video stream in Transport Stream, Element_Size may be unlimited
        Video_Unlimited=true;
        Element_Next=Buffer_Offset+6;
        PES_Packet_ElementSizes_Unlimited();
    }
    else
    {
        Element_Next=Buffer_Offset+Element_Size;

        //Testing end of the file before the end of the PES
        if (File_Offset+Element_Next>=File_Size) //This is the end of a truncated file
        {
            Element_Next=Buffer_Size;
            Element_Size=Element_Next-Buffer_Offset;
        }

        //Testing if we have enough data
        if (Element_Next>Buffer_Size)
            return false;
    }

    //Parsing header data
    if (!PES_Packet_Header_Data()) //Header
        return false;

    return true;
}

//---------------------------------------------------------------------------
// Packet header data
bool File_MpegPs::PES_Packet_Header_Data()
{
    TimeStamp.Have_PTS=false;
    TimeStamp.Have_DTS=false;

    //MPEG-2
    if (MPEG_Version==2)
    {
        if (Element_Size<3)
            return false;
        BitStream BS(Buffer+Buffer_Offset+Element_HeaderSize, Element_Size-6);
        BS.Skip(2);                                 //01
        BS.Skip(2);                                 //PES_scrambling_control
        BS.Skip(1);                                 //PES_priority
        BS.Skip(1);                                 //data_alignment_indicator
        BS.Skip(1);                                 //copyright
        BS.Skip(1);                                 //original_or_copy
        size_t PTS_DTS_flags=BS.Get(2);             //PTS_DTS_flags
        BS.Skip(1);                                 //ESCR_flag
        BS.Skip(1);                                 //ES_rate_flag
        BS.Skip(1);                                 //DSM_trick_mode_flag
        BS.Skip(1);                                 //additional_copy_info_flag
        BS.Skip(1);                                 //PES_CRC_flag
        BS.Skip(1);                                 //PES_extension_flag
        size_t PES_header_data_length=BS.Get(8);    //PES_header_data_length
        Element_HeaderSize+=3;
        if (6+3+PES_header_data_length>Element_Size)
            return false;

        //Options
        if (PTS_DTS_flags==0x2)
        {
            TimeStamp.Have_PTS=true;
            TimeStamp.Have_DTS=false;
            TimeStamp.PTS=MPEG_TimeStamp_Read(BS);
        }
        if (PTS_DTS_flags==0x3)
        {
            TimeStamp.Have_PTS=true;
            TimeStamp.Have_DTS=true;
            TimeStamp.PTS=MPEG_TimeStamp_Read(BS);
            TimeStamp.DTS=MPEG_TimeStamp_Read(BS);
        }

        //Positioning Packet data start
        Element_HeaderSize+=PES_header_data_length;
    }
    //MPEG-1
    else
    {
        //Keep out firsts 0xFF
        while (Buffer_Offset+Element_HeaderSize<Element_Next && Buffer[Buffer_Offset+Element_HeaderSize]==0xFF)
        {
            Element_HeaderSize++;
        }
        if (Buffer_Offset+Element_HeaderSize>=Element_Next)
        {
            Element_HeaderSize=Element_Size;
            return false;
        }

        //Buffer scale and size
        if ((Buffer[Buffer_Offset+Element_HeaderSize]&0xC0)==0x40)
        {
            Element_HeaderSize+=2;
        }

        //MPEG-1 with PTS
        if ((Buffer[Buffer_Offset+Element_HeaderSize]&0xF0)==0x20)
        {
            if (Element_Size<5)
                return false;
            BitStream BS(Buffer+Buffer_Offset+Element_HeaderSize, Element_Size);
            TimeStamp.Have_PTS=true;
            TimeStamp.Have_DTS=false;
            TimeStamp.PTS=MPEG_TimeStamp_Read(BS);
            Element_HeaderSize+=5;
        }

        //MPEG-1 with PTS and DTS
        else if ((Buffer[Buffer_Offset+Element_HeaderSize]&0xF0)==0x30)
        {
            if (Element_Size<10)
                return false;
            BitStream BS(Buffer+Buffer_Offset+Element_HeaderSize, Element_Size);
            TimeStamp.Have_PTS=true;
            TimeStamp.Have_DTS=true;
            TimeStamp.PTS=MPEG_TimeStamp_Read(BS);
            Element_HeaderSize+=10;
        }
    }
    return true;
}

//***************************************************************************
// Elements
//***************************************************************************

#define NAME(ELEMENT_NAME) \
    if (Config.Details_Get()) \
    { \
        Details_Add_Element(ELEMENT_NAME); \
    } \

#define INTEGRITY_GENERAL() \
    if (Count_Get(Stream_General)==0) \
        return; \

#define INTEGRITY(ELEMENT_SIZE) \
    if (Element_Size!=ELEMENT_SIZE) \
       return; \

//---------------------------------------------------------------------------
// Packet "BA"
void File_MpegPs::Start()
{
    NAME("Start")
    if (Count_Get(Stream_General)>0)
    {
        if (Config.Details_Get())
        {
            Details_Add_Element(_T("(Not needed)"));
        }
        return;
    }

    //Reading
    BitStream BS(Buffer+Buffer_Offset, Element_Size);
    size_t Version=BS.Get(2);                       //version
    if (Version==1)
    {
        //MPEG-2
        MPEG_Version=2;
        int8u Padding=Buffer[Buffer_Offset+9]&0x07;
        INTEGRITY(10+(size_t)Padding)
    }
    else
    {
        //MPEG-1
        MPEG_Version=1;
        INTEGRITY(8)
    }

    //Details
    if (Config.Details_Get())
    {
        Details_Add_Info(0, "Version", Version);
    }

    //Filling
    if (Count_Get(Stream_General)==0)
        Stream_Prepare(Stream_General);
}

//---------------------------------------------------------------------------
// Packet "BB"
void File_MpegPs::System()
{
    NAME("System")
    INTEGRITY_GENERAL()

    //Integrity
    if (Element_Size<2)
    {
        Element_Size=0;
        return;
    }

    //Integrity
    INTEGRITY((size_t)BigEndian2int16u(Buffer+Buffer_Offset)+2)            //header_lengh
}

//---------------------------------------------------------------------------
// Packet "BB"
void File_MpegPs::Padding()
{
    NAME("Padding")
}

//---------------------------------------------------------------------------
void File_MpegPs::Mpeg1a()
{
    NAME("Audio")

    //Mainly for TS streams in debug mode, which does not have Start chunk
    if (FromTS && Element_Name==0xC0 && General.empty())
        Stream_Prepare(Stream_General);

    //New stream if needed
    int8u Stream_Pos=Element_Name-0xC0;
    if (Stream_Pos>=Audio_Handle.size())
    {
        Audio_Handle.resize(Stream_Pos+1);
        Audio_TimeStamp.resize(Stream_Pos+1);
    }
    if (Audio_Handle[Stream_Pos]==NULL)
    {
        #if defined(MEDIAINFO_MPEGA_YES)
            Audio_Handle[Stream_Pos]=new File_Mpega;
            Audio_TimeStamp[Stream_Pos]=TimeStamp;
        #else
            //Filling
            Audio_Handle[Stream_Pos]=new File__Base();
            Audio_Handle[Stream_Pos]->Stream_Prepare(Stream_Audio);
            Audio_Handle[Stream_Pos]->Fill("Codec", "MPEG Audio");
        #endif
    }

    //Open Mpega
    Open_Buffer_Init(Audio_Handle[Stream_Pos], File_Size, File_Offset+Buffer_Offset);
    Open_Buffer_Continue(Audio_Handle[Stream_Pos], Buffer+Buffer_Offset, Element_Size);
}

//---------------------------------------------------------------------------
void File_MpegPs::Mpeg1v()
{
    NAME ("Video")

    if (!CompleteFileName.empty())
    {
        //INTEGRITY_GENERAL()
    }

    //Mainly for TS streams in debug mode, which does not have Start chunk
    if (FromTS && Element_Name==0xE0 && General.empty())
        Stream_Prepare(Stream_General);

    //New stream if needed
    int8u Stream_Pos=Element_Name-0xE0;
    if (Stream_Pos>=Video_Handle.size())
    {
        Video_Handle.resize(Stream_Pos+1);
        Video_Handle_Error.resize(Stream_Pos+1);
        Video_TimeStamp.resize(Stream_Pos+1);
        Video_TimeStamp_End.resize(Stream_Pos+1);
    }
    if (Video_Handle[Stream_Pos]==NULL)
    {
        #if defined(MEDIAINFO_MPEGV_YES)
            Video_Handle[Stream_Pos]=new File_Mpegv;
            ((File_Mpegv*)Video_Handle[Stream_Pos])->MPEG_Version=MPEG_Version;
            Video_TimeStamp[Stream_Pos]=TimeStamp;
        #else
            //Filling
            Video_Handle[Stream_Pos]=new File__Base();
            Video_Handle[Stream_Pos]->Stream_Prepare(Stream_Video);
            Video_Handle[Stream_Pos]->Fill("Codec", "MPEG Video");
        #endif
    }

    //TimeStamp
    Video_TimeStamp_End[Stream_Pos]=TimeStamp;

    //Open Mpegv
    Open_Buffer_Init(Video_Handle[Stream_Pos], File_Size, File_Offset+Buffer_Offset);
    Open_Buffer_Continue(Video_Handle[Stream_Pos], Buffer+Buffer_Offset, Element_Size);

    //MPEG-2 PS may transport MPEG-4V streams.
    if (Video_Handle_Error[Stream_Pos]==0 && MPEG_Version==2 && Video_Handle[Stream_Pos]->Count_Get(Stream_Video)==0)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Info(Error, "MPEG-V parsing was an error, testing MPEG4-V", 0);
        }

        #if defined(MEDIAINFO_MPEG4V_YES)
            File__Base* Mpeg4v=new File_Mpeg4v;
            Open_Buffer_Init(Mpeg4v, File_Size, File_Offset+Buffer_Offset);
            Open_Buffer_Continue(Mpeg4v, Buffer+Buffer_Offset, Element_Size);
            if (Mpeg4v->Count_Get(Stream_General)>0)
            {
                //This is MPEG-4V
                delete Video_Handle[Stream_Pos]; Video_Handle[Stream_Pos]=NULL;
                Video_Handle[Stream_Pos]=Mpeg4v;
            }
            else
            {
                delete Mpeg4v; Mpeg4v=NULL;
                Video_Handle_Error[Stream_Pos]=1;
            }
        #endif
    }
}

//---------------------------------------------------------------------------
void File_MpegPs::Private1()
{
    NAME("Private1")

    if (Element_Size<1)
        return;

    //Mainly for TS streams in debug mode, which does not have Start chunk
    if (FromTS && General.empty())
        Stream_Prepare(Stream_General);

    //Internal Stream_Pos
    int8u Private1_ID=Buffer[Buffer_Offset];
     //-Details
    if (Config.Details_Get())
    {
        Details_Add_Element(Ztring::ToZtring(Private1_ID, 16).MakeUpperCase());
    }

    //New stream if needed
   int8u Stream_Pos=Private1_ID;
    if (Stream_Pos>=Private1_Handle.size())
    {
        Private1_Handle.resize(Stream_Pos+1);
        Private1_TimeStamp.resize(Stream_Pos+1);
    }
    if (Private1_Handle[Stream_Pos]==NULL)
    {
        Private1_Handle[Stream_Pos]=Private1_ChooseParser(Stream_Pos);
    }

    //Open Mpegv
    Open_Buffer_Init(Private1_Handle[Stream_Pos], File_Size, File_Offset+Buffer_Offset);
    Open_Buffer_Continue(Private1_Handle[Stream_Pos], Buffer+Buffer_Offset, Element_Size);
}

//---------------------------------------------------------------------------
File__Base* File_MpegPs::Private1_ChooseParser(int8u Private1_ID)
{
    //Subtitles
         if (Private1_ID>=0x20 && Private1_ID<=0x3F)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Element("Text");
        }

        //Filling
        File__Base* Handle=new File__Base();
        Handle->Stream_Prepare(Stream_Text);
        Handle->Fill("Codec", "2-bit RLE");
        return Handle;
    }

    //AC3
    else if ((Private1_ID>=0x80 && Private1_ID<=0x87) || Private1_ID==0x0B) //0B for the beginning of a DTS stream (audio in MPEG-TS)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Element("AC3");
        }

        //Filling
        #if defined(MEDIAINFO_AC3_YES)
            return new File_Ac3();
        #else
            //Filling
            File__Base* Handle=new File__Base();
            Handle->Stream_Prepare(Stream_Audio);
            Handle->Fill("Codec", "AC3");
            return Handle;
        #endif
    }

    //DTS
    else if ((Private1_ID>=0x88 && Private1_ID<=0x8F) || Private1_ID==0x7F) //7F for the beginning of a DTS stream (audio in MPEG-TS)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Element("DTS");
        }

        //Filling
        #if defined(MEDIAINFO_DTS_YES)
            return new File_Dts();
        #else
            //Filling
            File__Base* Handle=new File__Base();
            Handle->Stream_Prepare(Stream_Audio);
            Handle->Fill("Codec", "DTS");
            return Handle;
        #endif
    }

    //PCM
    else if (Private1_ID>=0xA0 && Private1_ID<=0xA8)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Element("PCM");
        }

        //Filling
        File__Base* Handle=new File__Base();
        Handle->Stream_Prepare(Stream_Audio);
        Handle->Fill("Codec", "PCM");

        //Resolution
        int Resolution=Buffer[Buffer_Offset+5]>>6;
        if (Resolution==0) Handle->Fill("Resolution", 16);
        if (Resolution==1) Handle->Fill("Resolution", 20);
        if (Resolution==2) Handle->Fill("Resolution", 24);
        int SamplingRate=(Buffer[Buffer_Offset+5]&0x30)>>4;
        if (SamplingRate==0) Handle->Fill("SamplingRate", 48000);
        if (SamplingRate==1) Handle->Fill("SamplingRate", 96000);
        int Channels=Buffer[Buffer_Offset+5]&0x07;
        Handle->Fill("Channel(s)", Channels+1);
        return Handle;
    }

    //Unknown
    return NULL;
}

//---------------------------------------------------------------------------
void File_MpegPs::Private2()
{
    NAME("Private2")
}

//***************************************************************************
// Helpers
//***************************************************************************

//---------------------------------------------------------------------------
int64u File_MpegPs::MPEG_TimeStamp_Read (BitStream &BS)
{
    int64u TimeStamp;

    BS.Skip(4);
    TimeStamp=BS.Get(3);
    BS.Skip(1);
    TimeStamp<<=15;
    TimeStamp|=BS.Get(15);
    BS.Skip(1);
    TimeStamp<<=15;
    TimeStamp|=BS.Get(15);
    BS.Skip(1);
    return TimeStamp;
}

//---------------------------------------------------------------------------
bool File_MpegPs::Detect_NonMPEGPS ()
{
    //Element_Size
    if (Buffer_Size<8)
    {
        File_Offset=File_Size;
        return true;
    }

    //Detect DAT files, and the parser is not enough precise to detect them later
    if (CC4(Buffer)==CC4("RIFF"))
    {
        File_Offset=File_Size;
        return true;
    }

    //Detect TS files, and the parser is not enough precise to detect them later
    if (Buffer_Size>=188*4)
    {
        //Look for first Sync word
        while (Buffer_Offset<188 && CC1(Buffer+Buffer_Offset)!=0x47)
            Buffer_Offset++;
        if (Buffer_Offset<188 && CC1(Buffer+Buffer_Offset+188)==0x47 && CC1(Buffer+Buffer_Offset+188*2)==0x47 && CC1(Buffer+Buffer_Offset+188*3)==0x47)
        {
            File_Offset=File_Size;
            return true;
        }
        Buffer_Offset=0;
    }

    //Seems OK
    return false;
}

//---------------------------------------------------------------------------
void File_MpegPs::Detect_MPEG_Version ()
{
    size_t B5_Offset=Buffer_Offset;
    while (B5_Offset+4<Buffer_Size && CC3(Buffer+B5_Offset)!=0x000001 && CC1(Buffer+B5_Offset+1)!=0xB5)
        B5_Offset++;
    if (B5_Offset+4>=Buffer_Size)
        MPEG_Version=1;
    else
        MPEG_Version=2;
}

//---------------------------------------------------------------------------
void File_MpegPs::Detect_EOF_Needed ()
{
    //Jumping to the end of the file
    if (File_Size!=(int64u)-1 && File_Offset>=1024*1024 && File_Size>=File_Offset+Buffer_Size+4*1024*1024)
    {
        //Details
        if (Config.Details_Get())
        {
            Details_Add_Element(0, _T("-------------------------------------------"));
            Details_Add_Element(0, _T("---   MPEG-PS, Jumping to end of file   ---"));
            Details_Add_Element(0, _T("-------------------------------------------"));
        }

        //Jumping
        File_GoTo=File_Size-4*1024*1024;
    }
}

//---------------------------------------------------------------------------
size_t File_MpegPs::NextFrame ()
{
    if (!Video_Unlimited) //If Video unlimited, we stopped anywhere
    {
        //Look for first Sync word
        if (!FromTS)
        {
            while (Buffer_Offset+4<Buffer_Size && !(CC3(Buffer+Buffer_Offset)==0x000001 && CC1(Buffer+Buffer_Offset+3)>=0xB9)) //0xB9 is the lower value for MPEG-PS
                Buffer_Offset++;
        }
        else
        {
            while (Buffer_Offset+4<Buffer_Size && !(   CC3(Buffer+Buffer_Offset)==0x000001
                                                && (   CC1(Buffer+Buffer_Offset+3)==0xE0 //Video in TS
                                                    || CC1(Buffer+Buffer_Offset+3)==0xC0 //Audio in TS
                                                    || CC1(Buffer+Buffer_Offset+3)==0xBD //Private in TS
                                                )))
                Buffer_Offset++;
        }

        if (Buffer_Offset+4>=Buffer_Size)
            return Error;
    }

    return Buffer_Offset;
}

//---------------------------------------------------------------------------
bool File_MpegPs::PES_Packet_NextFrame ()
{
    //Keep out unuseful 0x00 in the stream
    while (Buffer_Offset+4<Buffer_Size && CC3(Buffer+Buffer_Offset)==0x000000)
        Buffer_Offset++;
    if (Buffer_Offset+4>=Buffer_Size)
        return false;

    //Look for Start code
    if (CC3(Buffer+Buffer_Offset)!=0x000001)
    {
        //Stream error. Try to find again the PS Pack start code to resynchronize
        while (Buffer_Offset+4<Buffer_Size && CC4(Buffer+Buffer_Offset)!=0x000001BA)
            Buffer_Offset++;
        if (Buffer_Offset+4>=Buffer_Size)
            return false; //No success to find again the PS Pack start code
    }

    Element_HeaderSize=4; //MPEG start code + Element name
    Element_Next=Buffer_Offset+4;
    Element_Size=4;
    return true;
}

bool File_MpegPs::PES_Packet_ElementName_IsOK()
{
    //Is it the start of a MPEG PS?
    if (Count_Get(Stream_General)==0
     && !(Element_Name>=0x20 && Element_Name<=0x3F)
     && !(Element_Name>=0x80 && Element_Name<=0x8F)
     &&   Element_Name!=0xBA
     && !(Element_Name>=0xBD && Element_Name<=0xEF)
     )
    {
        //MPEG PS didn't begin, skipping
        //Info
        Ztring Pos1; Pos1.From_Number(Element_Name, 16);
        Ztring Pos2;
        Pos2.resize(2-Pos1.size(), _T('0'));
        Pos2+=Pos1;
        Pos2.MakeUpperCase();
        Details_Add_Element(1, Pos2);
        Details_Add_Element(_T("Has not began, skipping"));
        PES_Packet_ElementSizes_Get();
        Buffer_Offset+=Element_Size;
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
bool File_MpegPs::PES_Packet_ElementSizes_Get()
{
    //PES Header
    if (Element_Name!=0xB9 //End
     && Element_Name!=0xBA //Start
     && Element_Name!=0xBB //System
     && Element_Name!=0xF0 //ECM
     && Element_Name!=0xF1 //EMM
     && Element_Name!=0xFF //Program stream directory
     && Element_Name!=0xF2 //DSMCC stram
     && Element_Name!=0xF8 //ITU-T Rec. H .222.1 type E
       )
    {
        if (!PES_Packet_Header()) //Header
            return false;
    }
    else
    {
        //Not possible to know the size directly
        //Searching Element_Next
        Element_Next=Buffer_Offset+4;
        while(Element_Next+3<=Buffer_Size && CC3(Buffer+Element_Next)!=0x000001)
            Element_Next++;
        if (Element_Next+3>Buffer_Size) //This is the last frame (truncated)
            return false;
        else if (CC2(Buffer+Element_Next)!=0x0000)
        {
            Element_Next++;
            if (CC1(Buffer+Element_Next)!=0x00)
                Element_Next++;
        }

        Element_Size=Element_Next-Buffer_Offset;
    }

    return true;
}

//---------------------------------------------------------------------------
void File_MpegPs::PES_Packet_ElementSizes_Unlimited()
{
    //The size is unlimited, searching the next Element_Name
    //Searching Element_Next
    while(Element_Next+4<=Buffer_Size && !(CC3(Buffer+Element_Next)==0x000001 && CC1(Buffer+Element_Next+3)==0xE0))
        Element_Next++;
    if (File_Offset+Element_Next+4>File_Size) //This is the end of file
        Element_Next+=3;
    else if (CC3(Buffer+Element_Next)!=0x000001)
    {
        Element_Next++;
        if (CC2(Buffer+Element_Next)!=0x0000)
        {
            Element_Next++;
            if (CC1(Buffer+Element_Next)!=0x00)
                Element_Next++;
        }
    }
    Element_Size=Element_Next-Buffer_Offset;
    Element_HeaderSize=0;
}

//***************************************************************************
// Information
//***************************************************************************

//---------------------------------------------------------------------------
void File_MpegPs::HowTo(stream_t StreamKind)
{
    switch (StreamKind)
    {
        case (Stream_General) :
            Fill_HowTo("Format", "R");
            Fill_HowTo("OveralBitRate", "R");
            Fill_HowTo("PlayTime", "R");
            Fill_HowTo("Encoded_Application", "R");
            break;
        case (Stream_Video) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("BitRate", "R");
            Fill_HowTo("Width", "R");
            Fill_HowTo("Height", "R");
            Fill_HowTo("AspectRatio", "R");
            Fill_HowTo("FrameRate", "R");
            Fill_HowTo("Delay", "R");
            break;
        case (Stream_Audio) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("BitRate", "R");
            Fill_HowTo("Channel(s)", "R");
            Fill_HowTo("SamplingRate", "R");
            Fill_HowTo("Resolution", "R");
            Fill_HowTo("Delay", "R");
            break;
        case (Stream_Text) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("Delay", "R");
            break;
        case (Stream_Chapters) :
            break;
        case (Stream_Image) :
            break;
        case (Stream_Menu) :
            break;
        case (Stream_Max) :
            break;
    }
}

} //NameSpace

#endif //MEDIAINFO_MPEGPS_YES

