// File__Base - Base for other files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Give common methods for all file types
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
#ifndef MediaInfo__BaseH
#define MediaInfo__BaseH
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/MediaInfo_Config.h"
#include <ZenLib/ZtringListList.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//***************************************************************************
// Class File__Base
//***************************************************************************

class File__Base
{
public :
    //Constructor/Destructor
    File__Base();
    virtual ~File__Base();

    //Analyze
    int     Open_File (const Ztring &CompleteFileName); //1 if succeed, 2 if problem with format, -1 if wrong type, -2 if no file
    void    Open_Buffer_Init (int64u File_Size=0, int64u File_Offset=0);
    void    Open_Buffer_Init (File__Base* Sub);
    void    Open_Buffer_Init (File__Base* Sub, int64u File_Size=0, int64u File_Offset=0);
    void    Open_Buffer_Continue (const int8u* Buffer, size_t Buffer_Size);
    void    Open_Buffer_Continue (File__Base* Sub, const int8u* Buffer, size_t Buffer_Size);
    void    Open_Buffer_Finalize ();
    void    Open_Buffer_Finalize (File__Base* Sub);
    int     Save ();

    //Get information
    Ztring  Inform ();
    Ztring  Inform (stream_t StreamKind, size_t StreamNumber=0); //All about only a specific stream
    const Ztring &Get (stream_t StreamKind, size_t StreamNumber, size_t Parameter, info_t KindOfInfo=Info_Text);
    const Ztring &Get (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, info_t KindOfInfo=Info_Text, info_t KindOfSearch=Info_Name);

    //Set information
    int           Set (stream_t StreamKind, size_t StreamNumber, size_t Parameter, const Ztring &ToSet, const Ztring &OldValue=_T(""));
    int           Set (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &ToSet, const Ztring &OldValue=_T(""));

    //Options
    size_t Count_Get (stream_t StreamKind, size_t Pos=Error) const;
    void   Language_Set (); //Update language for an existing File__Base

    //Infos
           ZtringListList Info_Capacities ();
    static ZtringListList Info_Capacities_Parameters ();

protected :
    //Theses classes need access to internal structure for optimization. There is recursivity with theses formats
    friend class File_Riff;
    friend class File_MpegPs;

    //Read
    virtual void Read_File            (); //To overload
    virtual void Read_Buffer_Init     (); //To overload
    virtual void Read_Buffer_Continue (); //To overload
    virtual void Read_Buffer_Finalize (); //To overload

    //Write
    virtual int Write       (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &ToSet, const Ztring &OldValue); //Write the value in memory
    virtual int WriteToDisk (); //Write modified tags on disk

    //How to
    virtual void HowTo (stream_t StreamType);

    //Before filling the stream, the stream must be prepared
    size_t Stream_Prepare   (stream_t KindOfStream);
    void   General_Fill     (); //Special : pre-fill General with some important information

    //Fill with datas
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const Ztring  &Value);
    void Fill_HowTo (stream_t StreamKind, size_t StreamPos, const char* Parameter, const char* Value);
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const std::string &Value, bool Utf8=true) {if (Utf8) Fill(StreamKind, StreamPos, Parameter, Ztring().From_UTF8(Value.c_str(), Value.size())); else Fill(StreamKind, StreamPos, Parameter, Ztring().From_Local(Value.c_str(), Value.size()));}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const char*    Value, intu Value_Size=Error, bool Utf8=true) {if (Utf8) Fill(StreamKind, StreamPos, Parameter, Ztring().From_UTF8(Value, Value_Size)); else Fill(StreamKind, StreamPos, Parameter, Ztring().From_Local(Value, Value_Size));}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const wchar_t* Value, intu Value_Size=Error) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Unicode(Value, Value_Size));}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int16u         Value, intu Radix=10) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int16s         Value, intu Radix=10) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int32u         Value, intu Radix=10) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int32s         Value, intu Radix=10) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int64u         Value, intu Radix=10) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int64s         Value, intu Radix=10) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    inline void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, float32        Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing) {Fill(StreamKind, StreamPos, Parameter, Ztring().From_Number(Value, AfterComma, Options));}
    inline void Fill (const char* Parameter, const Ztring  &Value) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value);} //With the last set
    inline void Fill (const char* Parameter, const std::string &Value, bool Utf8=true) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value);} //With the last set
    inline void Fill (const char* Parameter, const char*    Value, intu ValueSize=Error, bool Utf8=true) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, ValueSize, Utf8);} //With the last set
    inline void Fill (const char* Parameter, const int8u*   Value, intu ValueSize=Error, bool Utf8=true) {Fill(StreamKind_Last, StreamPos_Last, Parameter, (const char*)Value, ValueSize, Utf8);} //With the last set
    inline void Fill (const char* Parameter, const wchar_t* Value, intu ValueSize=Error) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, ValueSize);} //With the last set
    inline void Fill (const char* Parameter, int16u         Value, intu Radix=10) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int16s         Value, intu Radix=10) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int32u         Value, intu Radix=10) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int32s         Value, intu Radix=10) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int64u         Value, intu Radix=10) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int64s         Value, intu Radix=10) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, float32        Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, AfterComma, Options);} //With the last set
    inline void Fill (const char* Parameter, double         Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing) {Fill(StreamKind_Last, StreamPos_Last, Parameter, Value, AfterComma, Options);} //With the last set
    inline void Fill_HowTo (const char* Parameter, const char* Value) {Fill_HowTo(StreamKind_Last, StreamPos_Last, Parameter, Value);} //With the last set
    ZtringListList Fill_Temp;

    //Arrays
    std::vector<ZtringListList> General;
    std::vector<ZtringListList> Video;
    std::vector<ZtringListList> Audio;
    std::vector<ZtringListList> Text;
    std::vector<ZtringListList> Chapters;
    std::vector<ZtringListList> Image;
    std::vector<ZtringListList> Menu;
    std::vector<ZtringListList>* Stream[Stream_Max]; //pointer to others listed streams

    //Debug
    Ztring Details;
    void Details_Add_Element (int8u Level, Ztring Text);
    void Details_Add_Element (int8u Level, char* Text) {Details_Add_Element(Level, Ztring().From_UTF8(Text));}
    void Details_Add_Element (int8u Level, std::string Text) {Details_Add_Element(Level, Text.c_str());}
    void Details_Add_Element (const Ztring& Text); //More text about element
    void Details_Add_Element (const char* Text) {Details_Add_Element(Ztring().From_UTF8(Text));} //More text about element
    void Details_Add_Info (size_t Pos, const char* Parameter, const Ztring& Value);
    void Details_Add_Info (size_t Pos, const char* Parameter, const std::string& Value) {Details_Add_Info(Pos, Parameter, Ztring().From_Local(Value.c_str()));}
    void Details_Add_Info (size_t Pos, const char* Parameter, const char* Value, intu Value_Size=Error, bool Utf8=true) {if (Utf8) Details_Add_Info(Pos, Parameter, Ztring().From_UTF8(Value, Value_Size)); else Details_Add_Info(Pos, Parameter, Ztring().From_Local(Value, Value_Size));}
    void Details_Add_Info (size_t Pos, const char* Parameter, const int8u* Value, intu Value_Size=Error, bool Utf8=true) {Details_Add_Info(Pos, Parameter, (const char*)Value, Value_Size, Utf8);}
    void Details_Add_Info (size_t Pos, const char* Parameter, int16u Value, intu Radix=10) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    void Details_Add_Info (size_t Pos, const char* Parameter, int16s Value, intu Radix=10) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    void Details_Add_Info (size_t Pos, const char* Parameter, int32u Value, intu Radix=10) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    void Details_Add_Info (size_t Pos, const char* Parameter, int32s Value, intu Radix=10) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    void Details_Add_Info (size_t Pos, const char* Parameter, int64u Value, intu Radix=10) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    void Details_Add_Info (size_t Pos, const char* Parameter, int64s Value, intu Radix=10) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, Radix).MakeUpperCase());}
    void Details_Add_Info (size_t Pos, const char* Parameter, float32 Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, AfterComma, Options));}
    void Details_Add_Info (size_t Pos, const char* Parameter, double  Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing) {Details_Add_Info(Pos, Parameter, Ztring().From_Number(Value, AfterComma, Options));}
    void Details_Remove_Info ();
    int8u Details_Level_Base;
    int8u Details_Level_Last;

    //Optimization
    Ztring            CompleteFileName; //Cached filename for Read()
    std::vector<bool> Optimized[Stream_Max]; //If true, Arrays are not fully filled
    stream_t StreamKind_Last;
    size_t   StreamPos_Last;

    //Utils
    size_t Merge(const File__Base &ToAdd); //Merge 2 File_Base
    size_t Merge(File__Base &ToAdd, stream_t StreamKind, size_t StreamPos_From, size_t StreamPos_To); //Merge 2 streams

    //Divers
    void Clear();

public :
    //Buffer
    int8u* Buffer;
    size_t Buffer_Size;
    size_t Buffer_Size_Max;
    size_t Buffer_Offset; //Temporary usage in this parser
    size_t Buffer_MinimumSize;
    bool   Buffer_Init_Done;
    void Buffer_Clear();

    //File
    int64u File_Size;
    int64u File_Offset;
    int64u File_GoTo; //How many byte to skip?

    //End
    void Finalize();
    void Finalize__All      (stream_t StreamKind);
    void Finalize__All      (stream_t StreamKind, size_t Pos, Ztring &Codec_List, Ztring &Language_List);
    void Finalize_General   (size_t Pos);
    void Finalize_Video     (size_t Pos);
    void Finalize_Audio     (size_t Pos);
    void Finalize_Text      (size_t Pos);
    void Finalize_Chapters  (size_t Pos);
    void Finalize_Image     (size_t Pos);
    void Finalize_Menu      (size_t Pos);
    void Finalize_Tags      ();

public :  //A virer
    void Traiter(Ztring &C); //enleve les $if...
};

} //NameSpace

#endif
