// File_Dts - Info for DTS files
// Copyright (C) 2004-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Information about DTS files
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#include <MediaInfo/Setup.h>
#if defined(MEDIAINFO_DTS_YES)
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Audio/File_Dts.h"
#include <ZenLib/Utils.h>
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// DTS - Format
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//  StartCode                        4 bytes - 0x7FFE8001
//  Frame type                       1 bit
//  Deficit semple count?            5 bits
//  CRC present                      1 bit
//  Number of PCM sample block?      7 bits
//  Primary frame byte size?        14 bits
//  Channels                         6 bits
//  Sampling rate                    4 bits
//  Bit rate                         5 bits
//  ...                             14 bits
//  Resolution                       3 bits
//  ...
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
const size_t DTS_SamplingRate[]=
{     0,  8000,  16000,  32000,      0,      0,  11025,  22050,
  44100,     0,      0,  12000,  24000,  48000,  96000, 192000};
const size_t DTS_BitRate[]=
{   32000,   56000,   64000,   96000,  112000,  128000,  192000,  224000,
   256000,  320000,  384000,  448000,  512000,  576000,  640000,  768000,
   960000, 1024000, 1152000, 1280000, 1344000, 1408000, 1411200, 1472000,
  1536000, 1920000, 2048000, 3072000, 3840000};
const size_t DTS_Channels[]=
{1, 2, 2, 2, 2, 3, 3, 4,
 4, 5, 6, 6, 6, 7, 8, 8};
const size_t DTS_Resolution[]=
{16, 16, 20, 20, 24, 24, 0, 0};
const Char*  DTS_ChannelPositions[]=
{
    _T("Mono"),
    _T("Dual mono"),
    _T("L R"),
    _T("L R"),
    _T("L R"),
    _T("L C R"),
    _T("Front: L R, Surround: C"),
    _T("Front: L C R, Surround: C"),
    _T("Front: L R, Surround: L R"),
    _T("Front: L C R, Surround: L R"),
    _T("Front: L C C R, Surround: L R"),
    _T("Front: L C R, Rear: L R"),
    _T("Front: L R, Middle: L R, Rear: L R"),
    _T("Front: L C R, Middle: L R, surround: L R"),
    _T("Front: L R, Middle: L R, Surround: L C C R"),
    _T("Front: L C R, Middle: L R, Surround: L C R"),
};
//---------------------------------------------------------------------------

//***************************************************************************
// Format
//***************************************************************************

//---------------------------------------------------------------------------
void File_Dts::Read_Buffer_Init()
{
    Delay=0;
}

//---------------------------------------------------------------------------
void File_Dts::Read_Buffer_Continue()
{
    if (Buffer_Size<=4)
        return;
    if (Count_Get(Stream_Audio)>0)
    {
        File_Offset=File_Size;
        return; //Already done!
    }

    while (Buffer_Offset<=Buffer_Size-4 && !(Buffer[Buffer_Offset]==0x7F && Buffer[Buffer_Offset+1]==0xFE && Buffer[Buffer_Offset+2]==0x80 && Buffer[Buffer_Offset+3]==0x01))
        Buffer_Offset++;
    Delay+=Buffer_Offset;
    if (Buffer_Offset>Buffer_Size-4)
    {
        File_Offset=File_Size;
        return;
    }

    int64u I0=BigEndian2int64u((char*)Buffer+Buffer_Offset+4);
    size_t SamplingRate=(I0&0x000000003C000000LL)>>26;
    if (DTS_SamplingRate[SamplingRate]==0)
    {
        File_Offset=File_Size;
        return;
    }
    size_t BitRate     =(I0&0x0000000003E00000LL)>>21;
    size_t Channels    =(I0&0x0000000FC0000000LL)>>30;
    size_t Resolution  =(I0&0x0000000000000070LL)>> 4;

    //Fill stream
    Stream_Prepare(Stream_General);
    General[0](_T("Format"))=_T("DTS");
    General[0](_T("Format/String"))=_T("DTS");
    General[0](_T("Format/Extensions"))=_T("DTS");

    Stream_Prepare(Stream_Audio);
    Audio[0](_T("Codec"))=_T("DTS");
    //-SamplingRate
    Audio[0](_T("SamplingRate")).From_Number(DTS_SamplingRate[SamplingRate]);
    //-BitRate
    if (BitRate<29)
        Audio[0](_T("BitRate")).From_Number(DTS_BitRate[BitRate]);
    else if (BitRate==29)
        Audio[0](_T("BitRate"))=_T("Open");
    else if (BitRate==30)
        Audio[0](_T("BitRate"))=_T("Variable");
    else if (BitRate==31)
        Audio[0](_T("BitRate"))=_T("LossLess");
    if (BitRate<29 && CompleteFileName.empty())
        Audio[0](_T("Delay")).From_Number((float)Delay*1000/(DTS_BitRate[BitRate]/8), 0);
    //-Channels
    if (Channels<16)
    {
        Audio[0](_T("Channel(s)")).From_Number(DTS_Channels[Channels]+1);
        Audio[0](_T("ChannelPositions"))=DTS_ChannelPositions[Channels];
    }
    else
    {
        Audio[0](_T("Channel(s)")).From_Number(8);
        Audio[0](_T("ChannelPositions"))=_T("User defined");
    }
    Audio[0](_T("ChannelPositions"))+=_T(", Subwoofer");
    //-Resolution
    Audio[0](_T("Resolution")).From_Number(DTS_Resolution[Resolution]);
}

//---------------------------------------------------------------------------
void File_Dts::HowTo(stream_t StreamKind)
{
    switch (StreamKind)
    {
        case (Stream_General) :
            Fill_HowTo("Format", "R");
            break;
        case (Stream_Video) :
            break;
        case (Stream_Audio) :
            Fill_HowTo("Codec", "R");
            Fill_HowTo("BitRate", "R");
            Fill_HowTo("Channel(s)", "R");
            Fill_HowTo("ChannelPositions", "R");
            Fill_HowTo("SamplingRate", "R");
            Fill_HowTo("Resolution", "N");
            break;
        case (Stream_Text) :
            break;
        case (Stream_Chapters) :
            break;
        case (Stream_Image) :
            break;
        case (Stream_Menu) :
            break;
        case (Stream_Max) :
            break;
    }
}

} //NameSpace

#endif //MEDIAINFO_DTS_YES

