// Preferences - Manage preferences of MediaInfo
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Manage preferences of MediaInfo
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
//
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
#define NO_WIN32_LEAN_AND_MEAN
#include <vcl.h>
#pragma hdrstop
#include <Forms.hpp>
#include <Registry.hpp>
#include <Wininet.h>
#include <Shlobj.h>
#undef NO_WIN32_LEAN_AND_MEAN
#include "Preferences.h"
#include <wx/string.h>
#include <wx/dir.h>
#include <wx/url.h>
#include <wx/file.h>
#include <wx/filename.h>
#include <wx/thread.h>
#include <MediaInfo/MediaInfo.h>
//---------------------------------------------------------------------------
using namespace ZenLib;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// For Widgets with DEBUG in BCB
// Here only because this is useful in all MediaInfo
#ifdef _DEBUG
    void wxAssert(int, const wchar_t*, int, const wchar_t*, const wchar_t*) {}
#endif //_DEBUG
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
Preferences* Prefs=new Preferences;
//---------------------------------------------------------------------------

//***************************************************************************
// Constructor/Destructor
//***************************************************************************

//---------------------------------------------------------------------------
Preferences::Preferences()
{
    //Name of folders
    FolderNames=_T(
        "Language;"
        "Sheet;"
        "Tree;"
        "Custom;"
        "Language");
    //Name of default for kind of list
    DefaultNames=_T(
        "en;"
        "Example;"
        "Example;"
        "Example;"
        "en;");
}

//***************************************************************************
// Config
//***************************************************************************

//---------------------------------------------------------------------------
int Preferences::Config_Create()
{
    Config=_T(
        "Version;X.X.X.X\r\n"
        "NewestVersion;X.X.X.X\r\n"
        "Output;Basic\r\n"
        "Language;en\r\n"
        "Sheet;Example\r\n"
        "Tree;Example\r\n"
        "Custom;Example\r\n"
        "CheckUpdate;1\r\n"
        "ShellExtension;1\r\n"
        "ShowToolBar;1\r\n"
        "ShowMenu;1\r\n"
        "CloseAllAuto;1\r\n");
    Config(_T("Version")).FindAndReplace(_T("X.X.X.X"), MediaInfo_Version_GUI);
    Config(_T("NewestVersion")).FindAndReplace(_T("X.X.X.X"), MediaInfo_Version_GUI);
    Config.Save(BaseFolder+_T("MediaInfo.cfg"));

    return 1;
}

//---------------------------------------------------------------------------
int Preferences::Config_Load()
{
    int Retour=1;

    //Base folder
    BaseFolder=wxString(Application->ExeName.c_str(), wxConvLocal).c_str();
    BaseFolder=BaseFolder.substr(0, BaseFolder.rfind(_T("\\"))+1);
    if (!wxFileName::DirExists((BaseFolder+_T("Plugin\\")).c_str()))
        wxFileName::Mkdir((BaseFolder+_T("Plugin\\")).c_str());
    BaseFolder+=_T("Plugin\\");

    //Directories
    for (size_t Pos=0; Pos<FolderNames.size(); Pos++)
        if (!wxFileName::DirExists((BaseFolder+FolderNames[Pos]).c_str()))
            wxFileName::Mkdir((BaseFolder+FolderNames[Pos]).c_str());

    //Default language
    if (!wxFile::Exists((BaseFolder+_T("Language\\en.csv")).c_str()))
        Create(Language, _T("en"));
    Load(Language_English, _T("en"));

    //Configuration
    Config.Load(BaseFolder+_T("MediaInfo.cfg"));
    if (Config.size()==0 || Config(_T("Version"))!=MediaInfo_Version_GUI)
    {
        Config_Create();
        Retour=2;
    }

    RefreshFilesList(Language);
    Load(Language, Config(FolderNames[Language]));
    RefreshFilesList(Sheet);
    Load(Sheet, Config(FolderNames[Sheet]));
    RefreshFilesList(Tree);
    Load(Tree, Config(FolderNames[Tree]));
    RefreshFilesList(Custom);
    Load(Custom, Config(FolderNames[Custom]));

    //Shell Extension
    ExplorerShell();

    //Internet
    if (Retour!=2 && Config(_T("CheckUpdate"))==_T("1"))
        Prefs->InternetCheck();

    return Retour;
}

//---------------------------------------------------------------------------
int Preferences::Config_Save()
{
    return Config.Save();
}

//---------------------------------------------------------------------------
int Preferences::Config_Remove()
{
    return -1;
}

//***************************************************************************
// KindOfList management
//***************************************************************************

//---------------------------------------------------------------------------
int Preferences::Create(List_t KindOfList, const ZenLib::Ztring &Name)
{
    switch (KindOfList)
    {
        case Language :
            Details[KindOfList]=MediaInfoLib::MediaInfo::Option_Static(_T("Language_Get")).c_str();
            if (Name!=_T("en"))
                for (size_t Pos=0; Pos<Details[KindOfList].size(); Pos++)
                    Details[KindOfList](Pos, 1)=_T("");
            break;
        case Sheet :
            Details[KindOfList]=_T(
                "ColumnsCount;5\r\n"
                "Column0;General;0;CompleteName;30\r\n"
                "Column1;General;0;Format;10\r\n"
                "Column2;General;0;Video_Codec_List;10\r\n"
                "Column3;General;0;Audio_Codec_List;10\r\n"
                "Column4;General;0;Text_Codec_List;10");
            break;
        case Tree :
            break;
        case Custom :
            Details[KindOfList]=_T(
                "General;General           : %FileName%\\r\\nFormat            : %Format%$if(%OveralBitRate%, at %OveralBitRate/String%)\\r\\nLength            : %FileSize/String1% for %PlayTime/String1%\\r\\n\\r\\n\r\n"
                "Video;Video #%StreamKindID%          : %Codec/String%$if(%BitRate%, at %BitRate/String%)\\r\\nAspect            : %Width% x %Height% (%AspectRatio%) at %fps% fps\\r\\n\\r\\n\r\n"
                "Audio;Audio #%StreamKindID%          : %Codec/String%$if(%BitRate%, at %BitRate/String%)\\r\\nInfos             : %Channel(s)/String%, %SamplingRate/String%\\r\\n$if(%Language%,Language          : %Language%\\r\\n)\\r\\n\r\n"
                "Text;Text #%StreamKindID%           : %Codec/String%\\r\\n$if(%Language%,Language          : %Language%\\r\\n)\\r\\n\r\n"
                "Chapters;Chapters #%StreamKindID%       : %Total% chapters\\r\\n\\r\\n\r\n"
                "File_Begin\r\n"
                "File_End\r\n"
                "Page_Begin\r\n"
                "Page_Middle\r\n"
                "Page_End\r\n"
                "General_Begin\r\n"
                "General_End\r\n"
                "Video_Begin\r\n"
                "Video_Middle\r\n"
                "Video_End\r\n"
                "Audio_Begin\r\n"
                "Audio_Middle\r\n"
                "Audio_End\r\n"
                "Text_Begin\r\n"
                "Text_Middle\r\n"
                "Text_End\r\n"
                "Chapters_Begin\r\n"
                "Chapters_Middle\r\n"
                "Chapters_End");
            break;
        default:
            return -1;
    }
    Ztring Lang; if (KindOfList==Custom) Lang=Config(_T("Language"))+_T("."); //Special case : if Custom; we add Language in the file name
    Details[KindOfList].Save(BaseFolder+FolderNames[KindOfList]+_T("\\")+Lang+Name+_T(".csv"));

    return 1;
}

//---------------------------------------------------------------------------
int Preferences::Copy(List_t KindOfList, const ZenLib::Ztring &From, const ZenLib::Ztring &To)
{
    return wxCopyFile((BaseFolder+FolderNames[KindOfList]+_T("\\")+From+_T(".csv")).c_str(), (BaseFolder+_T("MediaInfo.")+FolderNames[KindOfList]+_T(".")+To+_T(".csv")).c_str());
}

//---------------------------------------------------------------------------
int Preferences::Load(List_t KindOfList, const ZenLib::Ztring &Name)
{
    //Test if the file exists
    if (!wxFile::Exists((BaseFolder+FolderNames[KindOfList]+_T("\\")+Name+_T(".csv")).c_str()))
        return -1;

    Details[KindOfList].Load(BaseFolder+FolderNames[KindOfList]+_T("\\")+Name+_T(".csv"));

    return 1;
}

//---------------------------------------------------------------------------
int Preferences::Remove(List_t KindOfList, const ZenLib::Ztring &Name)
{
    Ztring Lang; if (KindOfList==Custom) Lang=Config(_T("Language"))+_T("."); //Special case : if Custom; we add Language in the file name
    return wxRemoveFile((BaseFolder+FolderNames[KindOfList]+_T("\\")+Lang+Name+_T(".csv")).c_str());
}

//***************************************************************************
// FilesLists
//***************************************************************************

//---------------------------------------------------------------------------
int Preferences::RefreshFilesList(List_t KindOfList)
{
    //Default available?
    if (!wxFile::Exists((BaseFolder+FolderNames[KindOfList]+_T("\\")+DefaultNames[KindOfList]+_T(".csv")).c_str()))
        Create(KindOfList, DefaultNames[KindOfList]);

    //Listing
    wxDir Liste(BaseFolder.c_str());
    wxString filename;
    FilesList[KindOfList].clear();
    Ztring Lang;
    bool cont = Liste.GetFirst(&filename, (FolderNames[KindOfList]+_T("\\")+Lang+_T("*.csv")).c_str(), wxDIR_FILES);
    while (cont)
    {
        //Special case : Languages, should show the name of language in the local version
        if (KindOfList==Language)
        {
            if (FilesList[KindOfList].size()!=FilesList[Language_List].size())
                FilesList[Language_List].clear(); //FilesList[KindOfList] was resized to 0, must do the same thing
            ZtringListListF Lang;
            Lang.Load(BaseFolder+_T("Language\\")+filename.c_str());
            Ztring Local=Lang(_T("  Language_Name"));
            if (Local.size()==0)
                Local=Ztring(filename.c_str()).SubString(_T(""), _T(".csv"));
            FilesList[Language_List].push_back(Local);
        }

        //Add to the list
        FilesList[KindOfList].push_back(Ztring(filename.c_str()).SubString(_T(""), _T(".csv")).c_str());

        //Next one
        cont = Liste.GetNext(&filename);
    }

    if (FilesList[KindOfList].Find(Config(FolderNames[KindOfList]))==-1)
        Config(FolderNames[KindOfList])=DefaultNames[KindOfList];

    return 1;
}

//***************************************************************************
// System
//***************************************************************************

class ThreadInternetCheck:public TThread
{
public:
    __fastcall ThreadInternetCheck(bool CreateSuspended): TThread(CreateSuspended){}
    void __fastcall Execute();
};

void __fastcall ThreadInternetCheck::Execute()
{
    //Connexion test
    if (InternetGetConnectedState(NULL, 0)==0)
        return;// (void*)-1; //No internet connexion

    //Chargement du fichier sur le net
    wxURL H(wxString(_T("http://mediaarea.net/mediainfo_check/changelog_"))+MediaInfo_Version_GUI+_T(".bin"));
    wxInputStream *I=H.GetInputStream();
    if (!I)
        return;// (void*)-2;

    char A[1000];
    I->Read(&A, 1000);
    A[I->LastRead()]=0; //for the string
    delete I;
    ZtringListList Download(wxString(A, wxConvLocal).c_str());

    //Verification de la version
    Ztring Version=Prefs->Config.FindValue(_T("NewestVersion"), 1);
    Ztring NewestVersion=Download(_T("NewVersion"));
    if (NewestVersion!=_T("") && NewestVersion>Version)
    {
        //Affichage d'un message
        ZtringListList C3=Download.SubSheet(_T("NewMessage"));
        int Pos=C3.Find(Prefs->Config(_T("Language")), 1);
        if (Pos==-1)
            Pos=C3.Find(_T("en"), 1);
        if (Pos==-1)
            return;// (void*)-1;
        C3(Pos, 2).FindAndReplace(_T("\\r\\n"), _T("\r\n"));
        Application->MessageBox(AnsiString(C3(Pos, 2).c_str()).c_str(), "New version released!");
        //Inscription version connue pour pas repeter l'avertissement
        int Version_Pos=Prefs->Config.Find(_T("NewestVersion"));
        Prefs->Config(Version_Pos, 1)=NewestVersion;
        Prefs->Config.Save();
    }

    //Chargement de pages
    ZtringListList Pages=Download.SubSheet(_T("Url"));
    for (size_t Pos=0; Pos<Pages.size(); Pos++)
    {
        //Doit-on charger ou pas?
        if (Pages.Read(Pos, 1)==_T("1"))
        {
            std::wstring A=Pages.Read(Pos, 2).c_str();

            wxURL H2(Pages.Read(Pos, 2).c_str());
            wxInputStream *I=H2.GetInputStream();
            if (I)
            {
                char A[1000];
                I->Read(&A, 1000);
                delete I;
            }
        }
    }
    return;// (void*)1;
}

//---------------------------------------------------------------------------
int Preferences::InternetCheck()
{
    new ThreadInternetCheck(false);
    return 1;
}

//---------------------------------------------------------------------------
int Preferences::ExplorerShell()
{
    TRegistry* Reg=new TRegistry;
    Reg->RootKey = HKEY_CLASSES_ROOT;

    ZtringListList Liste;
    Liste=_T(
        ".mkv;MKVFile\r\n"
        ".mka;MKAFile\r\n"
        ".mks;MKSFile\r\n"
        ".ogg;OGGFile\r\n"
        ".ogm;OGMFile\r\n"
        ".wav;WAVFile\r\n"
        ".avi;AVIFile\r\n"
        ".divx;AVIFile\r\n"
        ".mpeg;mpegFile\r\n"
        ".mpg;mpegFile\r\n"
        ".dat;mpegFile\r\n"
        ".mpe;mpegFile\r\n"
        ".mpgx;mpegFile\r\n"
        ".mpm;mpegFile\r\n"
        ".m1s;mpegFile\r\n"
        ".vob;mpegFile\r\n"
        ".m2s;mpegFile\r\n"
        ".mp4;mpeg4File\r\n"
        ".m4v;mpeg4File\r\n"
        ".m4a;mpeg4File\r\n"
        ".mpgv;mpegFile\r\n"
        ".mpv;mpegFile\r\n"
        ".m1v;mpegFile\r\n"
        ".m2v;mpegFile\r\n"
        ".dat;mpegFile\r\n"
        ".mp2;mp3File\r\n"
        ".mp3;mp3File\r\n"
        ".mpc;mpcFile\r\n"
        ".mp+;mpcFile\r\n"
        ".asf;ASFFile\r\n"
        ".wmv;WMVFile\r\n"
        ".wma;WMAFile\r\n"
        ".mov;MOVFile\r\n"
        ".qt;QTFile\r\n"
        ".rm;RMFile\r\n"
        ".ra;RMFile\r\n"
        ".ifo;IfoFile\r\n"
        ".ac3;AC3File\r\n"
        ".dts;DTSFile\r\n"
        ".aac;AACFile\r\n"
        ".ape;APEFile\r\n"
        ".mac;APEFile\r\n"
        ".flac;FLACFile\r\n"
        ".3gp;mpeg4File\r\n"
        ".3gpp;mpeg4File\r\n"
        ".swf;FlashFile\r\n"
        ".flv;FlashVideoFile");

    bool IsChanged=false;
    for (size_t I1=0; I1<Liste.size(); I1++)
    {
        //Open (or create) a extension. Create only if Sheel extension is wanted
        if (Reg->OpenKey(Liste(I1, 0).c_str(), Config.Read(_T("ShellExtension")).To_int32s()))
        {
            //test if extension is known
            AnsiString Player=Reg->ReadString(_T(""));
            if (Player=="")
            {
                //extension not known, will use our default
                Player=Liste(I1, 1).c_str();
                try {Reg->WriteString(_T(""), Player);} catch (...){}
                IsChanged=true;
            }
            Reg->CloseKey();

            //Test if MediaInfo shell extension is known
            if (Reg->OpenKey(Player+_T("\\Shell\\Media Info\\Command"), false))
            {
                //MediaInfo shell extension is known
                if (Config.Read(_T("ShellExtension")).To_int32s())
                {
                    //test if good writing
                    AnsiString ShellExtensionToWtrite="\"" + Application->ExeName +"\" \"%1\"";
                    AnsiString ShellExtension=Reg->ReadString(_T("")).c_str();
                    if (ShellExtension!=ShellExtensionToWtrite)
                    {
                        //This is not the good shell extension, writing new one
                        try {Reg->WriteString(_T(""), ShellExtensionToWtrite);} catch (...){}
                        IsChanged=true;
                    }
                }
                else
                {
                    //Should not be here, deleting
                    Reg->CloseKey();
                    Reg->DeleteKey(Player+"\\Shell\\Media Info");
                    IsChanged=true;
                }
                Reg->CloseKey();
            }
            else
            {
                //MediaInfo Shell extension is not known
                if (Config.Read(_T("ShellExtension")).To_int32s())
                {
                    //Create it
                    Reg->OpenKey(Player+_T("\\Shell\\Media Info\\Command"), true);
                    AnsiString ShellExtensionToWtrite="\"" + Application->ExeName +"\" \"%1\"";
                    try {Reg->WriteString(_T(""), ShellExtensionToWtrite);} catch (...){}
                    Reg->CloseKey();
                    IsChanged=true;
                }
            }
        }
    }

    //end
    Reg->Free();
    if (IsChanged)
        SHChangeNotify(SHCNE_ASSOCCHANGED, SHCNF_IDLIST, NULL, NULL);

    return 1;
}

//***************************************************************************
// Translations
//***************************************************************************

//---------------------------------------------------------------------------
const ZenLib::Ztring &Preferences::Translate(const ZenLib::Ztring &Name)
{
    size_t Pos=Details[Language].Find(Name, 0, 0, L"==", Ztring_CaseSensitive);

    //If not in the language, search for English language
    if (Pos==-1 || Details[Language][Pos].size()<2)
        return Details[Language_English](Name);
    else
        return Details[Language](Pos)(1);
}

//---------------------------------------------------------------------------
ZenLib::Ztring Preferences::Translate(const ZenLib::Ztring &Count, const ZenLib::Ztring &Value)
{
    //Integrity
    if (Count.empty())
        return Translate(Value);

    size_t CountI=Count.To_int32u();
    size_t Pos3=CountI/100;
    int8u  Pos2=(CountI-Pos3)/10;
    int8u  Pos1=CountI-Pos3-Pos2;
    size_t Form=0;

    //Polish has 2 plurial, Algorithm of Polish
    if (Pos3==0)
    {
        if (Pos2==0)
        {
                 if (Pos1<=1)
                Form=1; //000 to 001 kanal
            else if (Pos1>=2 && Pos1<=4)
                Form=2; //002 to 004 kanaly
            else //if (Pos1>=5)
                Form=3; //005 to 009 kanalow
        }
        else if (Pos2==1)
                Form=3; //010 to 019 kanalow
        else //if (Pos2>=2)
        {
                 if (Pos1>=0 && Pos1<=1)
                Form=3; //020 to 021, 090 to 091 kanalow
            else if (Pos1>=2 && Pos1<=4)
                Form=2; //022 to 024, 092 to 094 kanali
            else //if (Pos1>=5)
                Form=3; //025 to 029, 095 to 099 kanalow
        }
    }
    else //if (Pos3>=1)
    {
        if (Pos2==0)
        {
                 if (Pos1<=1)
                Form=3; //100 to 101 kanalow
            else if (Pos1>=2 && Pos1<=4)
                Form=2; //102 to 104 kanaly
            else //if (Pos1>=5)
                Form=3; //105 to 109 kanalow
        }
        else if (Pos2==1)
                Form=3; //110 to 119 kanalow
        else //if (Pos2>=2)
        {
                 if (Pos1>=0 && Pos1<=1)
                Form=3; //120 to 121, 990 to 991 kanalow
            else if (Pos1>=2 && Pos1<=4)
                Form=2; //122 to 124, 992 to 994 kanali
            else //if (Pos1>=5)
                Form=3; //125 to 129, 995 to 999 kanalow
        }
    }

    Ztring ToReturn=Count;
         if (Form==1)
        ToReturn+=Translate(Value+_T("1"));
    else if (Form==2)
        ToReturn+=Translate(Value+_T("2"));
    else if (Form==3)
        ToReturn+=Translate(Value+_T("3"));
    return ToReturn;
}


