// File_Mpegv - Info for MPEG Video files
// Copyright (C) 2004-2005 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// File_Mpegv
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Information about MPEG Video files
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_MPEGV_YES) || (!defined(MEDIAINFO_VIDEO_NO) && !defined(MEDIAINFO_MPEGV_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Video/File_Mpegv.h"
#include "ZenLib/BitStream.h"
#include <ZenLib/Utils.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
const float Mpegv_FrameRate[]=
{
    0,
    23.976,
    24,
    25,
    29.97,
    30,
    50,
    59.94,
    60,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
};

const float Mpegv_Ratio1[]=
{
    0,
    1,
    0.6735,
    0.7031, //16/9 PAL
    0.7615,
    0.8055,
    0.8437, //16/9 NTSC
    0.8935,
    0.9157, //4/3 PAL
    0.9815,
    1.0255,
    1.0695,
    1.0950, //4/3 NTSC
    1.1575,
    1.2015,
    0,
};

const float Mpegv_Ratio2[]=
{
    0,
    1,
    (float)4/3,
    (float)16/9,
    2.21,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
};

const char* Mpegv_Chroma[]=
{
    "",
    "4:2:0",
    "4:2:2",
    "4:4:4",
};

const char* Mpegv_Standard[]=
{
    "Composite",
    "PAL",
    "NTSC",
    "SECAM",
    "MAC",
    "",
    "",
    "",
};

const char* Mpegv_PictureStructure[]=
{
    "Interlaced",
    "[TFF] Top field first",
    "[BFF] Bottom field first",
    "[PPF] Picture-per-field",
};

const char* Mpegv_Profile[]=
{
    "",
    "High",
    "Spatial",
    "SNR",
    "Main",
    "Simple",
    "",
    "",
};

const char* Mpegv_Level[]=
{
    "",
    "",
    "",
    "",
    "High",
    "",
    "High-1440",
    "",
    "Main",
    "",
    "Low",
    "",
    "",
    "",
    "",
    "",
};

//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

int File_Mpegv::Read(const int8u* Begin_, size_t Begin_Size_, const int8u* End_, size_t End_Size_, int64u FileSize)
{
    //Init
    Begin=Begin_;
    Begin_Size=Begin_Size_;
    End=End_;
    End_Size=End_Size_;
    Offset=0;

    //Integrity
    if (Begin_Size<4)
        return -1;
    if (CC4(Begin)==CC4("RIFF"))
        return -1; //Often in DAT files, and the parser is not enough presice to detect them

    MPEG_Time_Seconds=Error;
    MPEG_Time_Frames=Error;
    Library_Offset=Error;
    Chroma=Error;
    Standard=Error;
    PictureStructure=Error;
    Profile=Error;
    Level=Error;
    Offset_Next=0;
    ShouldStop=false;
    Corrupted=false;

    //Begin
    while (!ShouldStop)
    {
        Offset_Next+=4;
        while(Offset_Next+4<Begin_Size && CC3(Begin+Offset_Next)!=0x000001)
            Offset_Next++;
        if (Offset_Next+4>=Begin_Size)
            Offset_Next=Begin_Size; //This is the last frame
        Size=Offset_Next-Offset;
        if (Size<=4 || CC3(Begin+Offset)!=0x000001) ;
        else if (CC1(Begin+Offset+3)==0x00) Picture_Start();
        else if (CC1(Begin+Offset+3)>=0x01
              && CC1(Begin+Offset+3)<=0xAF) Slice_Start();
        else if (CC1(Begin+Offset+3)==0xB2) User_Start();
        else if (CC1(Begin+Offset+3)==0xB3) Sequence_Header();
        else if (CC1(Begin+Offset+3)==0xB4) Sequence_Error();
        else if (CC1(Begin+Offset+3)==0xB5) Extension_Start();
        else if (CC1(Begin+Offset+3)==0xB7) Sequence_End();
        else if (CC1(Begin+Offset+3)==0xB8) Group_Start(Begin);
        else if (CC1(Begin+Offset+3)>=0xB9) System_Start();
        //Go after the last frame
        Offset+=Size;
        while (Offset<Begin_Size && Begin[Offset]==0x00)
            Offset++;
        if (Offset>=Begin_Size)
            ShouldStop=true;
        else if (CC3(Begin+Offset-2)!=0x000001)
            return -1; // No right bytes found
        Offset-=2;
    }

    if (General.empty() && Video.empty())
        return -1;

    //End
    size_t MPEG_BeginTime_Seconds=Error;
    size_t MPEG_BeginTime_Frames=Error;
    size_t MPEG_EndTime_Seconds=Error;
    size_t MPEG_EndTime_Frames=Error;
    if (End_Size>128)
    {
        MPEG_BeginTime_Seconds=MPEG_Time_Seconds;
        MPEG_BeginTime_Frames =MPEG_Time_Frames;
        MPEG_Time_Seconds=Error;
        MPEG_Time_Frames=Error;
        Offset=End_Size-4;
        while (Offset>0 && !(CC3(End+Offset)==0x000001 && End[Offset+3]==0xB8))
            Offset--;
        if (Offset>0)
        {
            Group_Start(End);

            //Go after the last frame
            Offset+=Size;
            while (Offset<End_Size && End[Offset]==0x00)
                Offset++;
            if (Offset>=End_Size)
                ; //ShouldStop=true;
            else if (CC3(End+Offset-2)!=0x000001)
                return -1; // No right bytes found
            Offset-=2;
        }
        MPEG_EndTime_Seconds=MPEG_Time_Seconds;
        MPEG_EndTime_Frames =MPEG_Time_Frames;
    }


    //Some calculations
    //-Detect MPEG version
    //-TODO: Warning: Expect MPEG-1 stream in MPEG-1 container, and MPEG-2 stream in MPEG-2 container. if container not known, try a workaround
    if (MPEG_Version==0)
    {
        //MPEG version not known, try to detect it
        if (Width>400)
            MPEG_Version=2;
        else
            MPEG_Version=1;
    }
    //AspectRatio
    float AspectRatio=0;
    if (MPEG_Version==2)
    {
        if (RatioValue==1 && Height!=0)
            AspectRatio=Width/Height;
        else
            AspectRatio=Mpegv_Ratio2[RatioValue];
    }
    else
    {
        if (Height!=0)
            AspectRatio=(float)Width/Height*Mpegv_Ratio1[RatioValue];;
    }

    //Filling
    if (MPEG_Version==2)
    {
        Fill(Stream_General, 0, "Format", "MPV2");
        Fill(Stream_Visual, 0, "Codec", "MPEG-2");
        Fill(Stream_Visual, 0, "Codec/String", "MPEG-2 Video");
    }
    else
    {
        Fill(Stream_General, 0, "Format", "MPV1");
        Fill(Stream_Visual, 0, "Codec", "MPEG-1");
        Fill(Stream_Visual, 0, "Codec/String", "MPEG-1 Video");
        PictureStructure=3;
    }
    if (Library_Offset!=Error)
        Fill(Stream_Visual, 0, "Encoded_Library", (const char*)(Begin+Library_Offset), Library_Size);
    Fill(Stream_Visual, 0, "Width", Width);
    Fill(Stream_Visual, 0, "Height", Height);
    Fill(Stream_Visual, 0, "AspectRatio", AspectRatio);
    Fill(Stream_Visual, 0, "FrameRate", Mpegv_FrameRate[FrameRate]);

    if (BitRate==0x3FFFF)
        Fill(Stream_Visual, 0, "BitRate_Mode", "VBR");
    else
    {
        Fill(Stream_Visual, 0, "BitRate_Mode", "CBR");
        Fill(Stream_Visual, 0, "BitRate", BitRate*400);
    }

    if (Chroma<4)
        Fill(Stream_Visual, 0, "Chroma", Mpegv_Chroma[Chroma]);
    if (Standard<8)
        Fill(Stream_Visual, 0, "Standard", Mpegv_Standard[Standard]);
    if (PictureStructure<4)
        Fill(Stream_Visual, 0, "Interlacement", Mpegv_PictureStructure[PictureStructure]);
    if (Profile<8 && Level<16)
        Fill(Stream_Visual, 0, "Profile", Ztring().From_Local(Mpegv_Profile[Profile])+_T("@")+Ztring().From_Local(Mpegv_Level[Level]));

    //Calculate InitTime with Framerate
    if (MPEG_EndTime_Seconds!=Error)
    {
        if (MPEG_BeginTime_Seconds==Error)
            MPEG_BeginTime_Seconds=0; //Not found, expect this is 0...
        size_t MPEG_BeginTime=MPEG_BeginTime_Seconds*1000;
        size_t MPEG_EndTime =MPEG_EndTime_Seconds*1000;
        if (!Video.empty() && !Video[0](_T("FrameRate")).empty())
        {
            int32u FrameRate=Video[0](_T("FrameRate")).To_int32s();
            if (FrameRate)
            {
                MPEG_BeginTime+=MPEG_BeginTime_Frames*1000/FrameRate;
                MPEG_EndTime  +=MPEG_EndTime_Frames  *1000/FrameRate;
            }
        }
        Video[0](_T("PlayTime")).From_Number(MPEG_EndTime-MPEG_BeginTime);
    }

    return 1;
}

//---------------------------------------------------------------------------
// Packet "00", Picture Start
//
void File_Mpegv::Picture_Start()
{
}

//---------------------------------------------------------------------------
// Packet "01" --> "AF", Slice Start
//
void File_Mpegv::Slice_Start()
{
}

//---------------------------------------------------------------------------
// Packet "B2", User Start
//
void File_Mpegv::User_Start()
{
    if (Video.empty())
        return;

    Library_Offset=Offset;
    Library_Size=Size;
}

//---------------------------------------------------------------------------
// Packet "B3", Sequence Header
//
void File_Mpegv::Sequence_Header()
{
    if (!General.empty())
        return;
    Stream_Prepare(Stream_General);
    Fill("Format", "MPEG-1/2 Video");
    Stream_Prepare(Stream_Visual);
    Fill("Codec", "MPEG-1/2 Video");

    BitStream BS(Begin+Offset, Size);
    BS.Skip(32);                                    //header
    Width=BS.Get(12);                               //horizontal_size_value
    Height=BS.Get(12);                              //vertical_size_value
    RatioValue=BS.Get(4);                           //aspect_ratio_information
    FrameRate=BS.Get(4);                            //frame_rate_code
    BitRate=BS.Get(18);                             //bit_rate_value
    BS.Skip(1);                                     //marker_bit
    BS.Skip(10);                                    //vbv_buffer_size_value
    BS.Skip(1);                                     //constrained_parameters_flag
    int32u Intra   =BS.Get(1);                      //load_intra_quantiser_matrix
    if (Intra)
        {for (size_t Pos=0; Pos<8; Pos++) {BS.Skip(32); BS.Skip(32);}}; //intra_quantiser_matrix[64]
    int32u Intra_No=BS.Get(1);                      //load_non_intra_quantiser_matrix
    if (Intra_No)
        {for (size_t Pos=0; Pos<8; Pos++) {BS.Skip(32); BS.Skip(32);}}; //non_intra_quantiser_matrix[64]

    //Coherancy test
    BS.Byte_Align();
    Size=BS.Offset_Get();
}

//---------------------------------------------------------------------------
// Packet "B4", Sequence Error
//
void File_Mpegv::Sequence_Error()
{
}

//---------------------------------------------------------------------------
// Packet "B5", Extension Start
//
void File_Mpegv::Extension_Start()
{
    if (Video.empty())
        return;

    BitStream BS(Begin+Offset, Size);
    BS.Skip(32);                                    //header
    size_t ID=BS.Get(4);                            //extension_start_code_identifier

    //sequence_extension
         if (ID==1)
    {
        BS.Skip(1);                                 //profile_and_level_indication, escape
        Profile=BS.Get(3);                          //profile_and_level_indication, profile
        Level=BS.Get(4);                            //profile_and_level_indication, level
        size_t Progressive=BS.Get(1);               //progressive_sequence
        Chroma=BS.Get(2);                           //chroma_format
        size_t Width_Ext=BS.Get(2);                 //horizontal_size_extension
        size_t Height_Ext=BS.Get(2);                //vertical_size_extension
        size_t BitRate_Ext=BS.Get(12);              //bit_rate_extension
        BS.Skip(1);                                 //marker_bit
        BS.Skip(8);                                 //vbv_buffer_size_extension
        BS.Skip(1);                                 //low_delay
        size_t FrameRate_N=BS.Get(2);               //frame_rate_extension_n
        size_t FrameRate_D=BS.Get(5);               //frame_rate_extension_d

        //Coherancy test
        BS.Byte_Align();
        Size=BS.Offset_Get();

        //Filling
        Width+=0x1000*Width_Ext;
        Height+=0x1000*Height_Ext;
        BitRate+=0x4000*BitRate_Ext;
        if (FrameRate_D)
            FrameRate=FrameRate_N/FrameRate_D;
        if (Progressive)
            PictureStructure=3;
        else
            PictureStructure=0;
    }
    //sequence_display_extension
    else if (ID==2)
    {
        Standard=BS.Get(3);                         //video_format
        if (BS.Get(1))                              //colour_description
        {
            BS.Skip(8);                             //colour_primaries
            BS.Skip(8);                             //transfer_characteristics
            BS.Skip(8);                             //matrix_coefficients
        }
        BS.Skip(14);                                //display_horizontal_size
        BS.Skip(1);                                 //marker_bit
        BS.Skip(14);                                //display_vertical_size

        //Coherancy test
        BS.Byte_Align();
        Size=BS.Offset_Get();
    }
    //
    else if (ID==3)
    {
    }
    //
    else if (ID==4)
    {
    }
    //
    else if (ID==5)
    {
    }
    //
    else if (ID==6)
    {
    }
    //
    else if (ID==7)
    {
    }
    //
    else if (ID==8)
    {
        BS.Skip(4);                                 //f_code[0][0] (forward horizontal)
        BS.Skip(4);                                 //f_code[0][1] (forward vertical)
        BS.Skip(4);                                 //f_code[1][0] (backward horizontal)
        BS.Skip(4);                                 //f_code[1][1] (backward vertical)
        BS.Skip(2);                                 //intra_dc_precision
        BS.Skip(2);                                 //picture_structure
        size_t TopField=BS.Get(1);                  //top_field_first
        BS.Skip(1);                                 //frame_pred_frame_dct
        BS.Skip(1);                                 //concealment_motion_vectors
        BS.Skip(1);                                 //q_scale_type
        BS.Skip(1);                                 //intra_vlc_format
        BS.Skip(1);                                 //alternate_scan
        BS.Skip(1);                                 //repeat_first_field
        BS.Skip(1);                                 //chroma_420_type
        size_t Progressive=BS.Get(1);               //progressive_frame
        if (BS.Get(1))                              //composite_display_flag
        {
            BS.Skip(1);                             //v_axis
            BS.Skip(3);                             //field_sequence
            BS.Skip(1);                             //sub_carrier
            BS.Skip(7);                             //burst_amplitude
            BS.Skip(8);                             //sub_carrier_phase
        }

        //Coherancy test
        BS.Byte_Align();
        Size=BS.Offset_Get();

        //Filling
        if (Progressive)   PictureStructure=3; //Progressive
        else if (TopField) PictureStructure=1; //TopField
        else               PictureStructure=2; //BottomField
    }
    //
    else if (ID==9)
    {
    }
    //
    else if (ID==10)
    {
    }
}

//---------------------------------------------------------------------------
// Packet "B7", Sequence_End
//
void File_Mpegv::Sequence_End()
{
}

//---------------------------------------------------------------------------
// Packet "B8", Group Start
//
void File_Mpegv::Group_Start(const int8u* Begin)
{
    if (MPEG_Time_Seconds!=Error || MPEG_Time_Frames!=Error)
        return; //We want only the first one

    BitStream BS(Begin+Offset, Size);
    BS.Skip(32);                                    //header
    BS.Skip(1);                                     //time_code, drop_frame_flag
    size_t Hours  =BS.Get(5);                       //time_code, time_code_hours
    size_t Minutes=BS.Get(6);                       //time_code, time_code_minutes
    BS.Skip(1);                                     //time_code, marker_bit
    size_t Seconds=BS.Get(6);                       //time_code, time_code_seconds
    size_t Frames =BS.Get(6);                       //time_code, time_code_pictures
    BS.Skip(1);                                     //closed_gop
    BS.Skip(1);                                     //broken_link

    //Coherancy test
    BS.Byte_Align();
    Size=BS.Offset_Get();

    //Some additional calculations
    MPEG_Time_Seconds=60*60*Hours+60*Minutes+Seconds;
    MPEG_Time_Frames =Frames;
}

//---------------------------------------------------------------------------
// Packet "B9" --> "FF" , System_Start
//
void File_Mpegv::System_Start()
{
}

//---------------------------------------------------------------------------
void File_Mpegv::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("Encoded_Application"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Visual)
    {
        Video[0](_T("Codec"), Info_HowTo)=_T("R");
        Video[0](_T("BitRate"), Info_HowTo)=_T("R");
        Video[0](_T("Width"), Info_HowTo)=_T("R");
        Video[0](_T("Height"), Info_HowTo)=_T("R");
        Video[0](_T("AspectRatio"), Info_HowTo)=_T("R");
        Video[0](_T("FrameRate"), Info_HowTo)=_T("R");
    }
}

} //NameSpace

#endif //MEDIAINFO_MPEGV_*

