// File_Avc - Info for AVC Video files
// Copyright (C) 2006-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_MPEG4V_YES) || (!defined(MEDIAINFO_VIDEO_NO) && !defined(MEDIAINFO_MPEG4V_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Video/File_Mpeg4v.h"
#include "ZenLib/BitStream.h"
#include "ZenLib/Utils.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
const char* Mpeg4v_PictureStructure[]=
{
    "Interlaced",
    "[TFF] Top field first",
    "[BFF] Bottom field first",
    "[PPF] Picture-per-field",
};

const char* Mpeg4v_Chroma[]=
{
    "",
    "4:2:0",
    "",
    "",
};
//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

//---------------------------------------------------------------------------
File_Mpeg4v::File_Mpeg4v()
{
    IVOP_Count=0;
    PVOP_Count=0;
    BVOP_Count=0;
    SVOP_Count=0;
    NVOP_Count=0;
    QPel=0;
    GMC=0;
    interlaced=Error;
    newpred_enable=Error;
    time_size=Error;
    reduced_resolution_vop_enable=Error;
    shape=Error;
    sprite_enable=Error;
    scalability=Error;
    enhancement_type=Error;
    complexity_estimation_disable=Error;
}

//---------------------------------------------------------------------------
int File_Mpeg4v::Read(const int8u* Begin_, size_t Begin_Size_, const int8u* End_, size_t End_Size_, int64u FileSize)
{
    //Init
    Begin=Begin_;
    Begin_Size=Begin_Size_;
    End=End_;
    End_Size=End_Size_;
    Offset=0;

    //Integrity
    if (Begin_Size<4)
        return -1;
    if (CC4(Begin)==CC4("RIFF"))
        return -1; //Often in DAT files, and the parser is not enough presice to detect them

    Library_Offset=Error;
    Chroma=Error;
    Standard=Error;
    PictureStructure=Error;
    Offset_Next=0;
    ShouldStop=false;
    while (!ShouldStop)
    {
        Offset_Next+=4;
        while(Offset_Next+4<Begin_Size && CC3(Begin+Offset_Next)!=0x000001)
            Offset_Next++;
        if (Offset_Next+4>=Begin_Size)
            Offset_Next=Begin_Size; //This is the last frame
        Size=Offset_Next-Offset;
        if (Size<=4 || CC3(Begin+Offset)!=0x000001) ;
        else if (CC1(Begin+Offset+3)<=0x1F) VideoObject_Start();
        else if (CC1(Begin+Offset+3)>=0x20
              && CC1(Begin+Offset+3)<=0x2F) VideoObjectLayer_Start();
        else if (CC1(Begin+Offset+3)==0xB0) VisualObjectSequence_Start();
        else if (CC1(Begin+Offset+3)==0xB1) VisualObjectSequence_End();
        else if (CC1(Begin+Offset+3)==0xB2) User_Start();
        else if (CC1(Begin+Offset+3)==0xB3) GroupOfVOP_Start();
        else if (CC1(Begin+Offset+3)==0xB5) VisualObject_Start();
        else if (CC1(Begin+Offset+3)==0xB6) VideoObjectPlane_Start();
        //Go after the last frame
        Offset+=Size;
        while (Offset<Begin_Size && Begin[Offset]==0x00)
            Offset++;
        if (Offset>=Begin_Size)
            ShouldStop=true;
        else if (CC3(Begin+Offset-2)!=0x000001)
            return -1; // No right bytes found
        Offset-=2;
    }

    if (General.empty() && Video.empty())
        return -1;

    if (Library_Offset!=Error)
        Fill(Stream_Visual, 0, "Encoded_Library", (const char*)(Begin+Library_Offset), Library_Size);
    if (Chroma<4)
        Fill(Stream_Visual, 0, "Chroma", Mpeg4v_Chroma[Chroma]);
    if (PictureStructure<4)
        Fill(Stream_Visual, 0, "Interlacement", Mpeg4v_PictureStructure[PictureStructure]);
    Ztring Codec_Settings;
    if (BVOP_Count>0)   Codec_Settings+=_T("BVOP ");
    if (NVOP_Count>0)   Codec_Settings+=_T("NVOP ");
    if (GMC)            Codec_Settings+=_T("GMC")+Ztring::ToZtring(GMC)+_T(" ");
    if (QPel)           Codec_Settings+=_T("QPel ");
    if (!Codec_Settings.empty())
    {
        Codec_Settings.resize(Codec_Settings.size()-1);
        Fill(Stream_Visual, 0, "Codec_Settings", Codec_Settings);
    }

    return 1;
}

//---------------------------------------------------------------------------
// Packet "00"
//
void File_Mpeg4v::VideoObject_Start()
{
}

//---------------------------------------------------------------------------
// Packet "20"
//
void File_Mpeg4v::VideoObjectLayer_Start()
{
    Stream_Prepare(Stream_General);
    Fill("Format", "MPEG-4 Video");
    Stream_Prepare(Stream_Visual);
    Fill("Codec", "MPEG-4 Video");

    size_t Width=0;
    size_t Height=0;
    size_t PixelAspectRatio_Width=0;
    size_t PixelAspectRatio_Height=0;

    BitStream BS(Begin+Offset, Begin_Size-Offset);
    BS.Skip(32);				                    //header
    BS.Skip(1);				                        //random_accessible_vol
    BS.Skip(8);				                        //object_type_id
    size_t VerId=1;
    if (BS.Get(1))                                  //is_object_layer_id
    {
        VerId=BS.Get(4);			                //object_layer_verid
        BS.Skip(3);				                    //object_layer_priority
    }
    size_t PixelAspectRatio=BS.Get(4);              //aspect_ratio_info
    if (PixelAspectRatio==0x0F)
    {
        PixelAspectRatio_Width=BS.Get(8);           //par_width
        PixelAspectRatio_Height=BS.Get(8);          //par_height
    }
    if (BS.Get(1))                                  //vol_control_parameters
    {
        Chroma=BS.Get(2);		                    //chroma_format
        BS.Skip(1);				                    //low_delay
        if (BS.Get(1))                              //vbv_parameters
        {
            BS.Skip(15);				            //first_half_bit_rate
            BS.Skip(1);				                //marker_bit
            BS.Skip(15);				            //latter_half_bit_rate
            BS.Skip(1);				                //marker_bit
            BS.Skip(15);				            //first_half_vbv_buffer_size
            BS.Skip(1);				                //marker_bit
            BS.Skip(3);				                //latter_half_vbv_buffer_size
            BS.Skip(11);				            //first_half_vbv_occupancy
            BS.Skip(1);				                //marker_bit
            BS.Skip(15);				            //latter_half_vbv_occupancy
            BS.Skip(1);				                //marker_bit
        }
    }
    shape=BS.Get(2);                         //object_layer_shape
    size_t Shape_Extension=0;
    if (shape==3 && VerId!=1) //Shape=GrayScale
        Shape_Extension=BS.Get(4);				    //object_layer_shape_extension
    size_t aux_comp_count=0;
    if (Shape_Extension==0 && Shape_Extension==1 && Shape_Extension==5 && Shape_Extension==7 && Shape_Extension==8) aux_comp_count=1;
    if (Shape_Extension==2 && Shape_Extension==3 && Shape_Extension==6 && Shape_Extension==9 && Shape_Extension==11) aux_comp_count=2;
    if (Shape_Extension==4 && Shape_Extension==10 && Shape_Extension==12) aux_comp_count=3;
    BS.Skip(1);				                        //marker_bit
    size_t Increment=BS.Get(16);                    //vop_time_increment_resolution
    size_t PowerOf2=1;
    for (time_size=0; time_size<=16; time_size++)
    {
        if (Increment<PowerOf2)
            break;
        PowerOf2<<=1;
    }
    BS.Skip(1);				                        //marker_bit
    if (BS.Get(1))                                  //fixed_vop_rate
        BS.Skip(time_size);                         //fixed_vop_time increment
    size_t Resolution=8;
    if (shape!=2) //Shape!=BinaryOnly
    {
        if (shape==0) //Shape=Rectangular
        {
            BS.Skip(1);				                //marker_bit
            Width=BS.Get(13);	                    //object_layer_width
            BS.Skip(1);				                //marker_bit
            Height=BS.Get(13);                      //object_layer height
            BS.Skip(1);				                //marker_bit
        }
        interlaced=BS.Get(1);                       //interlaced
        if (interlaced)
            PictureStructure=0;
        else
            PictureStructure=3;
        BS.Skip(1);                                 //obmc_disable
        if (VerId==1)
            sprite_enable=BS.Get(1);                //sprite_enable
        else
            sprite_enable=BS.Get(2);                //sprite_enable
        if (sprite_enable==1 || sprite_enable==2 )  //static or GMC
        {
            if (sprite_enable!=2) //No GMC
            {
                BS.Skip(13);	                    //sprite_width
                BS.Skip(1);				            //marker_bit
                BS.Skip(13);                        //sprite_height
                BS.Skip(1);				            //marker_bit
                BS.Skip(13);	                    //sprite_top_coordinate
                BS.Skip(1);				            //marker_bit
                BS.Skip(13);                        //sprite_left_coordinate
                BS.Skip(1);				            //marker_bit
            }
            GMC=BS.Get(6);                          //no_of_sprite_warping_points
            BS.Skip(2);                             //sprite_warping_accuracy
            BS.Skip(1);                             //sprite_brightness_change
            if (sprite_enable!=2) //No GMC
                BS.Skip(1);                         //low_latency_sprite_enable
        }
        if (VerId==1 && shape!=0) //Shape!=Rectangular
            BS.Get(1);                              //sadct_disable
        if (BS.Get(1))                              //bits_per_pixel not_8_bit
        {
            BS.Skip(4);                             //quant_precision
            Resolution=BS.Get(4);                   //bits_per_pixel
        }
        if (shape==3) //Shape=GrayScale
        {
            BS.Skip(1);                             //no_gray_quant_update
            BS.Skip(1);                             //composition_method
            BS.Skip(1);                             //linear_composition
        }
        size_t quant_type=BS.Get(1);                //quant_type
        if (quant_type)
        {
            size_t load_intra_quant_mat=BS.Get(1);  //load_intra_quant_mat
            if (load_intra_quant_mat)
            {
                size_t Pos;
                for (Pos=0; Pos<64; Pos++)
                    if (BS.Get(8)==0)                //intra_quant_mat
                        Pos=Error;
                if (Pos==64)
                    return; //there is a problem
            }
            size_t load_nonintra_quant_mat=BS.Get(1);//load_nonintra_quant_mat
            if (load_nonintra_quant_mat)
            {
                size_t Pos;
                for (Pos=0; Pos<64; Pos++)
                    if (BS.Get(8)==0)                //nonintra_quant_mat
                        Pos=Error;
                if (Pos==64)
                    return; //there is a problem
            }
            if(shape==3) //Shape=GrayScale
            {
                for(size_t Pos=0; Pos<aux_comp_count; Pos++)
                {
                    size_t load_intra_quant_mat_grayscale=BS.Get(1);//load_intra_quant_mat_grayscale
                    if (load_intra_quant_mat_grayscale)
                    {
                        size_t Pos;
                        for (Pos=0; Pos<64; Pos++)
                            if (BS.Get(8)==0)        //intra_quant_mat_grayscale
                                Pos=Error;
                        if (Pos==64)
                            return; //there is a problem
                    }
                    size_t load_nonintra_quant_mat_grayscale=BS.Get(1);//load_nonintra_quant_mat_grayscale
                    if (load_nonintra_quant_mat_grayscale)
                    {
                        size_t Pos;
                        for (Pos=0; Pos<64; Pos++)
                            if (BS.Get(8)==0)        //nonintra_quant_mat_grayscale
                                Pos=Error;
                        if (Pos==64)
                            return; //there is a problem
                    }
                }
            }
        }
        if (VerId!=1)
            QPel=BS.Get(1);                         //quarter_sample
        complexity_estimation_disable=BS.Get(1);    //complexity_estimation_disable
        if (complexity_estimation_disable)
            return; //TODO: parse it, but this value is not common
        BS.Skip(1);                                 //resync_marker_disable
        if (BS.Get(1)==1)                           //data_partitioned
            BS.Skip(1);                             //reversible_vlc
        if (VerId!=1)
        {
            newpred_enable=BS.Get(1);               //newpred_enable
            if (newpred_enable)
            {
                BS.Skip(2);                         //requested_upstream_message_type
                BS.Skip(1);                         //newpred_segment_type
            }
            reduced_resolution_vop_enable=BS.Get(1);//reduced_resolution_vop_enable
        }
        scalability=BS.Get(1);                      //scalability
        if (scalability==1)
        {
            size_t Hierarchy=BS.Get(1);             //hierarchy_type
            BS.Skip(4);                             //ref_layer_id
            BS.Skip(1);                             //ref_layer_sampling_direc
            BS.Skip(5);                             //hor_sampling_factor_n
            BS.Skip(5);                             //hor_sampling_factor_m
            BS.Skip(5);                             //vert_sampling_factor_n
            BS.Skip(5);                             //vert_sampling_factor_m
            enhancement_type=BS.Get(1);             //enhancement_type
            if (shape==1 && Hierarchy==0) //Shape=Binary
            {
                BS.Skip(1);                         //use_ref_shape
                BS.Skip(1);                         //use_ref_texture
                BS.Skip(5);                         //shape_hor_sampling_factor_n
                BS.Skip(5);                         //shape_hor_sampling_factor_m
                BS.Skip(5);                         //shape_vert_sampling_factor_n
                BS.Skip(5);                         //shape_vert_sampling_factor_m
            }
        }
    }
    else
    {
        if (VerId!=1)
        {
            if (BS.Get(1)==1)                       //scalability
            {
                BS.Skip(4);                         //ref_layer_id
                BS.Skip(5);                         //shape_hor_sampling_factor_n
                BS.Skip(5);                         //shape_hor_sampling_factor_m
                BS.Skip(5);                         //shape_vert_sampling_factor_n
                BS.Skip(5);                         //shape_vert_sampling_factor_m
            }
        }
        BS.Skip(1);                                 //resync_marker_disable
    }

    //Coherancy test
    BS.Byte_Align();
    Size=BS.Offset_Get();

    //Filling
    if (Height)
    {
        Fill("Width", Width);
        Fill("Height", Height);
        float PixelAspectRatio_Value=1;
             if (PixelAspectRatio==0x01) PixelAspectRatio=1;
             if (PixelAspectRatio==0x02) PixelAspectRatio=12/11;
        else if (PixelAspectRatio==0x03) PixelAspectRatio=10/11;
        else if (PixelAspectRatio==0x04) PixelAspectRatio=16/11;
        else if (PixelAspectRatio==0x05) PixelAspectRatio=40/13;
        else if (PixelAspectRatio==0x0F && PixelAspectRatio_Height) PixelAspectRatio_Value=((float)PixelAspectRatio_Width)/PixelAspectRatio_Height;
        Fill("AspectRatio", ((float)Width)/Height*PixelAspectRatio_Value);
    }
        Fill("Resolution", Resolution);
}

//---------------------------------------------------------------------------
// Packet "B0"
void File_Mpeg4v::VisualObjectSequence_Start()
{
}

//---------------------------------------------------------------------------
// Packet "B1"
void File_Mpeg4v::VisualObjectSequence_End()
{
}

//---------------------------------------------------------------------------
// Packet "B2", User defined size, this is often used of library name
// Data                             X bytes, Pos=0
//
void File_Mpeg4v::User_Start()
{
    if (Video.empty())
        return;

    Library_Offset=Offset;
    Library_Size=Size;
}

//---------------------------------------------------------------------------
// Packet "B3"
void File_Mpeg4v::GroupOfVOP_Start()
{
}

//---------------------------------------------------------------------------
// Packet "B5"
void File_Mpeg4v::VisualObject_Start()
{
}

//---------------------------------------------------------------------------
// Packet "B6"
//
void File_Mpeg4v::VideoObjectPlane_Start()
{
    if (Video.empty())
        return;

    BitStream BS(Begin+Offset, Begin_Size-Offset);
    BS.Skip(32);				                    //header
    size_t vop_coding_type=BS.Get(2);               //vop_coding_type
    size_t modulo_time_base=0;
    while (BS.Get(1)==1)
        modulo_time_base++;                         //modulo_time_base
    BS.Skip(1);                                     //marker_bit
    BS.Get(time_size);                              //time_increment
    BS.Skip(1);                                     //marker_bit
    size_t vop_coded=BS.Get(1);                     //vop_coding
    if (vop_coded)
    {
        if (newpred_enable)
        {
            BS.Skip(time_size);                     //vop_id
            if (BS.Get(1)==1)                       //vop_id_for_prediction_indication
                BS.Skip(time_size);                 //vop_id_for_prediction
            BS.Skip(1);                             //marker_bit
        }

        if (shape!=2              //Shape!=BinaryOnly
         && (vop_coding_type==1   //Type=P
          || (vop_coding_type==3  //Type=S
           && sprite_enable==2))) //Sprite_Enable=GMC
            BS.Skip(1);
        if (reduced_resolution_vop_enable==1
         && shape==0                         //Shape=Rectangular
         && (vop_coding_type==1              //Type=P
          || vop_coding_type==0))            //Type=I
            BS.Skip(1);
        if (shape!=0) //Shape!=Rectangular
        {
            if (sprite_enable==1    //Sprite_Enable=Static
             && vop_coding_type==0) //Type=I
            {
                BS.Skip(13);	                    //vop_width
                BS.Skip(1);				            //marker_bit
                BS.Skip(13);                        //vop_height
                BS.Skip(1);				            //marker_bit
                BS.Skip(13);	                    //vop_horizontal_mc_spatial_ref
                BS.Skip(1);				            //marker_bit
                BS.Skip(13);                        //vop_vertical_mc_spatial_ref
                BS.Skip(1);				            //marker_bit
            }
            if (shape!=2             //Shape=BinaryOnly
             && scalability==1
             && enhancement_type==1)
                BS.Skip(1);                         //background_composition
            BS.Skip(1);                             //change_conv_ratio_disable
            if (BS.Get(1)==1)                       //vop_constant_alpha
                BS.Skip(8);                         //vop_constant_alpha_value
        }

        if (shape!=2) //Shape=BinaryOnly
            if (complexity_estimation_disable==0)
                return; //TODO: parse it, but this value is not common
        if (shape!=2) //Shape=BinaryOnly
        {
            BS.Skip(3);                             //intra_dc_vlc_thr
            if (interlaced)
            {
                if (BS.Get(1)==1)                   //top_field_first
                    PictureStructure=1; //Top first
                else
                    PictureStructure=2; //Bottom first
                BS.Skip(1);                         //alternate_vertical_scan_flag
            }
        }
        //...
    }

    if (!vop_coded)              NVOP_Count++; //VOP with no data
    else if (vop_coding_type==0) IVOP_Count++; //Type I
    else if (vop_coding_type==1) PVOP_Count++; //Type P
    else if (vop_coding_type==2) BVOP_Count++; //Type B
    else if (vop_coding_type==3) SVOP_Count++; //Type S
}

//---------------------------------------------------------------------------
void File_Mpeg4v::HowTo()
{
}

} //NameSpace

#endif //MEDIAINFO_MPEG4V_*
