// File_Riff - Info for RIFF files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_RIFF_YES) || (!(defined(MEDIAINFO_VIDEO_NO) && defined(MEDIAINFO_AUDIO_NO)) && !defined(MEDIAINFO_RIFF_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
    #pragma hdrstop
#endif
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#include "MediaInfo/Multiple/File_Riff.h"
#include "MediaInfo/Video/File_Mpeg4v.h"
#include "MediaInfo/Audio/File_Mpega.h"
#include "MediaInfo/Audio/File_Ac3.h"
#include <ZenLib/Utils.h>
#include <ZenLib/Ztring.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
extern MediaInfo_Config Config;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//To clarify the code
//-Begin
#define BEGIN \
while (!ShouldStop && Offset<Element_Offset[Level]+8+Element_Size[Level] && Offset+12<Begin_Size) \
{ \
    Level++; \
    Element_Offset[Level]=Offset; \
    Element_Size  [Level]=LittleEndian2int32u((const char*)Begin+Offset+4); \
    if (Element_Size[Level]%2) Element_Size  [Level]++; /* Pad byte */ \
    Element_List  [Level]=CC4(Begin+Offset+8); \
    if (0) \
    {

//-For each Chunk: (ChunkName) (Size) (ChunkDatas)
#define ATOM(_ATOM) \
    } \
    else if (CC4(Begin+Offset)==CC4(_ATOM) && Element_Offset[Level]+12+Element_Size[Level]<Begin_Size) \
    { \
        Offset+=8;

//-For each Chunk, call a deafault function
#define ATOM_DEFAULT \
    } \
    else if (Element_Offset[Level]+12+Element_Size[Level]<Begin_Size) \
    { \
        const char* Parameter=(const char*)(Begin+Offset); \
        Offset+=8;

//-For each List: LIST (size) (ChunkName) (ListOfChunks)
#define LIST(_LIST) \
    } \
    else if ((CC4(Begin+Offset)==CC4("LIST") || CC4(Begin+Offset)==CC4("RIFF")) && CC4(Begin+Offset+8)==CC4(_LIST)) \
    { \
        Offset+=12; \

//-End
#define END \
    } \
    Offset=Element_Offset[Level]+8+Element_Size[Level]; \
    Level--; \
    if (Offset>=Begin_Size) \
        ShouldStop=true; \
}

//---------------------------------------------------------------------------
int File_Riff::Read(const int8u* Begin_, size_t Begin_Size_, const int8u* End_, size_t End_Size_, int64u FileSize)
{
    //Init
    Begin=Begin_;
    Begin_Size=Begin_Size_;
    End=End_;
    End_Size=End_Size_;
    Offset=0;

    //Integrity test
    if (Begin_Size<32 || CC4(Begin)!=CC4("RIFF"))
        return -1;

	//Init
	ShouldStop=false;
    Level=0;
    for (size_t Pos=0; Pos<10; Pos++) Element_Offset[Pos]=0;
    for (size_t Pos=0; Pos<10; Pos++) Element_Size  [Pos]=0;
    for (size_t Pos=0; Pos<10; Pos++) Element_List  [Pos]=0; //4CC of the list type
    TotalFrame=0; //Count of frame in all AVI file (with odml too)
    FrameRate=0;
    Avih_Offset=Error;
    bool Movi_Rec_Present=false;

    //Hierarchy
    BEGIN
    LIST("AVI ") Stream_Prepare(Stream_General); Fill("Format", "AVI");
        BEGIN
        LIST("hdrl") //Header
            BEGIN
            ATOM("avih") Avih_Analyse();
            LIST("strl") Strl_Analyse();
                BEGIN
                ATOM("strh") Strh_Analyse();
                ATOM("strf") Strf_Analyse();
                ATOM("strd") //Datas
                ATOM("strn") Strn_Analyse();
                ATOM("indx") //Index
                ATOM("vprp") //Video property header
				ATOM("JUNK") Junk_Analyse();
                END
            LIST("odml") //OpenDML
                BEGIN
                ATOM("dmlh") Dmlh_Analyse();
                END
			ATOM("JUNK") Junk_Analyse();
            ATOM_DEFAULT Info_Analyse(Parameter);
            END
        LIST("INFO")
            BEGIN
            ATOM_DEFAULT Info_Analyse(Parameter);
            END
        LIST("exif")
            BEGIN
            ATOM_DEFAULT Exif_Analyse(Parameter);
            END
        LIST("movi") //<- Aligned on sector - 12
            BEGIN
            LIST("rec ")
                Movi_Rec_Present=true; //Some players can not read them
                BEGIN
                ATOM_DEFAULT Movi_Analyse(Parameter);
                END
            ATOM_DEFAULT Movi_Analyse(Parameter);
            END
        ATOM("JUNK") Junk_Analyse();
        ATOM("CSET") //Cset_Analyse();
        ATOM("idx1") //Index
        END
    LIST("AVIX") //OpenDML
        LIST("movi")
            BEGIN
            LIST("rec ")
                BEGIN
                ATOM_DEFAULT Movi_Analyse(Parameter);
                END
            ATOM_DEFAULT Movi_Analyse(Parameter);
            END
        ATOM("idx1") //Index
    LIST("PAL ") Stream_Prepare(Stream_General); Fill("Format", "RIFF Palette");
    LIST("RDIB") Stream_Prepare(Stream_General); Fill("Format", "RIFF DIB");
    LIST("RMID") Stream_Prepare(Stream_General); Fill("Format", "RIFF MIDI");
    LIST("RMMP") Stream_Prepare(Stream_General); Fill("Format", "RIFF MMP");
    LIST("WAVE") Stream_Prepare(Stream_General); Fill("Format", "Wave");
        BEGIN
        ATOM("fmt ") Strf_Auds_Analyse();
        ATOM("fact") //Fact_Analyse();
        //This atom is too large, but we need only first bytes
        }
        else if (CC4(Begin+Offset)==CC4("data") && Element_Offset[Level]+12<Begin_Size)
        {
            Offset+=8;
            Element_Size[Level]=Begin_Size-Element_Offset[Level]-12;
            Movi_Analyse("00wb"); //Raw datas
        END
    END

    //Is it OK?
    if (General.empty())
        return -1;

    //Some work on the first video stream
    if (!Video.empty())
    {
        if (TotalFrame!=0)
            Fill(Stream_Visual, 0, "FrameCount", TotalFrame);

        int64u PlayTime=0;
        if (FrameRate!=0)
        {
            PlayTime=(int64u)(TotalFrame/FrameRate*1000);
            Fill(Stream_General, 0, "PlayTime", PlayTime);
        }

        //Calculation of missing information
        if (PlayTime!=0)
        {
            int32u OveralBitRate=(int32u)(General[0](_T("FileSize")).To_float32()/PlayTime*8*1000);
            Fill(Stream_General, 0, "OveralBitRate", OveralBitRate);
            if (OveralBitRate>100000) //OveralBitRate is > 100 000, to avoid strange behavior
            {
                int32u VideoBitRate=OveralBitRate-5000; //5000 bps because of AVI OverHead (from GordianKnot)
                for (size_t Pos=0; Pos<Audio.size(); Pos++)
                {
                    int OverHead=8000; //Classic OverHead (for MP3) (from GordianKnot)
                    if (Audio[Pos](_T("Codec"))==_T("2000")) //AC3
                        OverHead=4750; //OverHead of AC3 in AVI (from GordianKnot)
                    VideoBitRate-=Audio[Pos](_T("BitRate")).To_int32s()+OverHead;
                }
                Fill(Stream_Visual, 0, "BitRate", VideoBitRate);
            }
        }
    }

    return 1;
}

//---------------------------------------------------------------------------
// Chunk "avih", 56 bytes
// MicrosecPerFrame                 4 bytes, Pos=0
// MaxBytesPerSec                   4 bytes, Pos=4
// PaddingGranularity               4 bytes, Pos=8
// Flags                            4 bytes, Pos=12
// TotalFrames                      4 bytes, Pos=16 (Only for the first AVI chunk)
// InitialFrames                    4 bytes, Pos=20
// StreamsCount                     4 bytes, Pos=24
// SuggestedBufferSize              4 bytes, Pos=28
// Width                            4 bytes, Pos=32
// Height                           4 bytes, Pos=36
// Reserved                         16 bytes, Pos=40
//
// Flags:
// Has index                        0x00000010
// Must use index                   0x00000020
// Is interleaved                   0x00000100
// Use CK type to find key frames   0x00000800
// Was capture file                 0x00010000
// Copyrighted                      0x00020000
//
void File_Riff::Avih_Analyse()
{
    //Coherancy test
    if (Element_Size[Level]<56) //Size of atom
        return;

    //Save info for later use
    Avih_Offset=Offset;
    TotalFrame=LittleEndian2int32u(Begin+Offset+16); //TotalFrames
}

//---------------------------------------------------------------------------
// Chunk "CSET", 8 bytes
// CodePage                         2 bytes, Pos=0
// CountryCode                      2 bytes, Pos=2
// LanguageCode                     2 bytes, Pos=4
// Dialect                          2 bytes, Pos=6
//
void File_Riff::Cset_Analyse()
{
    //TODO: take a look about IBM/MS RIFF/MCI Specification 1.0
}

//---------------------------------------------------------------------------
// Chunk "dmlh", 4 bytes for fccType "auds"
// GrandFrames                      2 bytes, Pos=0
//
void File_Riff::Dmlh_Analyse()
{
    TotalFrame=LittleEndian2int32u(Begin+Offset+0); //GrandFrames
}

//---------------------------------------------------------------------------
// List of Exif atoms
// Name                             X bytes, Pos=0
//
void File_Riff::Exif_Analyse(const char* Parameter)
{
    //Coherancy test
    if (Count_Get(Stream_General)==0)
        return;

    if (0) ;
    else if (CC4(Parameter)==CC4("ecor")) Fill(Stream_General, 0, "Make", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("emdl")) Fill(Stream_General, 0, "Model", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("emnt")) Fill(Stream_General, 0, "MakerNotes", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("erel")) Fill(Stream_General, 0, "RelatedImageFile", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("etim")) Fill(Stream_General, 0, "Written_Date", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("eucm")) Fill(Stream_General, 0, "Comment", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ever")) ; //Exif version
    else Fill(Stream_General, 0, Ztring().From_Local(Parameter, 4).To_Local().c_str(), (const char*)(Begin+Offset), Element_Size[Level]);
}

//---------------------------------------------------------------------------
// Chunk "fact", at least bytes
// SamplesCount                     2 bytes, Pos=0
//
void File_Riff::Fact_Analyse()
{
}

//---------------------------------------------------------------------------
// List of information atoms
// Name                             X bytes, Pos=0
//
void File_Riff::Info_Analyse(const char* Parameter)
{
    //Coherancy test
    if (Count_Get(Stream_General)==0)
        return;

    //List all possibilities one per one
    if (0) ;
    else if (CC4(Parameter)==CC4("IARL")) Fill(Stream_General, 0, "Archival_Location", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IART")) Fill(Stream_General, 0, "Artist", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ICMS")) Fill(Stream_General, 0, "CommissionedBy", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ICMT")) Fill(Stream_General, 0, "Comment", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ICOP")) Fill(Stream_General, 0, "Copyright", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ICRD")) Fill(Stream_General, 0, "Written_Date", Ztring().Date_From_String((const char*)(Begin+Offset), Element_Size[Level]));
    else if (CC4(Parameter)==CC4("ICRP")) Fill(Stream_General, 0, "Cropped", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IDIM")) Fill(Stream_General, 0, "Dimensions", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IDIT")) Fill(Stream_General, 0, "Mastered_Date", Ztring().Date_From_String((const char*)(Begin+Offset), Element_Size[Level]));
    else if (CC4(Parameter)==CC4("IDPI")) Fill(Stream_General, 0, "DotsPerInch", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IENG")) Fill(Stream_General, 0, "Engineer", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IGNR")) Fill(Stream_General, 0, "Genre", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IKEY")) Fill(Stream_General, 0, "Keywords", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ILGT")) Fill(Stream_General, 0, "Ligthness", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IMED")) Fill(Stream_General, 0, "Medium", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("INAM")) Fill(Stream_General, 0, "Title", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IPLT")) Fill(Stream_General, 0, "NumColors", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("IPRD")) Fill(Stream_General, 0, "Product", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ISBJ")) Fill(Stream_General, 0, "Subject", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ISFT")) Fill(Stream_General, 0, "Encoded_Application", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ISHP")) Fill(Stream_General, 0, "Sharpness", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ISRC")) Fill(Stream_General, 0, "Encoded_Original/DistributedBy", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ISRF")) Fill(Stream_General, 0, "Encoded_Original", (const char*)(Begin+Offset), Element_Size[Level]);
    else if (CC4(Parameter)==CC4("ITCH")) Fill(Stream_General, 0, "EncodedBy", (const char*)(Begin+Offset), Element_Size[Level]);
    else Fill(Stream_General, 0, Ztring().From_Local(Parameter, 4).To_Local().c_str(), (const char*)(Begin+Offset), Element_Size[Level]);
}

//---------------------------------------------------------------------------
// Chunk "JUNK", library defined size for padding, often used to store library name
// Junk                             X bytes, Pos=0
//
void File_Riff::Junk_Analyse()
{
    //Coherancy test
    if (Element_Size[Level]<8 || Count_Get(Stream_General)==0)
        return;

    //Detect DivX files
         if (CC5(Begin+Offset)==CC5("DivX "))
        Fill(Stream_General, 0, "Format", "DivX");
    //MPlayer
    else if (CC8(Begin+Offset)==CC8("[= MPlay") && Get(Stream_General, 0, _T("Encoded_Library")).empty())
        Fill(Stream_General, 0, "Encoded_Library", "MPlayer");
    //Other libraries?
    else if (CC1(Begin+Offset)>=CC1("A") && CC1(Begin+Offset)<=CC1("z") && Get(Stream_General, 0, _T("Encoded_Library")).empty())
        Fill(Stream_General, 0, "Encoded_Library", (const char*)(Begin+Offset), Element_Size[Level]);
}

//---------------------------------------------------------------------------
// Analyse Movi chuncks.
// Can be :
// ..wb     Audio
// ..dc     Video
// ..tx     Text
//
void File_Riff::Movi_Analyse(const char* Parameter)
{
    //Initialize external ressources
    if (Codec_External.empty())
    {
        //TODO: a better handling of PosInStream, BETA
        Codec_External.resize(256);
        for (size_t Pos=0; Pos<Codec_External.size(); Pos++)
            Codec_External[Pos]=NULL;
    }
    int32u Avi_Stream_Pos=Ztring().From_Local(Parameter, 2).To_int32u();
    Ztring Avi_Stream_Kind; Avi_Stream_Kind.From_Local(Parameter, 2, 2);
    if (1)//Avi_Stream_Pos==0)
    {
        if (Avi_Stream_Kind==_T("dc"))
        {
            Ztring Codec=Get(Stream_Visual, 0, _T("Codec"));
            //MPEG-4
            if (Config.Codec_Get(Get(Stream_Visual, 0, _T("Codec")), InfoCodec_KindofCodec).find(_T("MPEG-4"))==0)
            {
                //We want to analyse all chunks
                if (!Codec_External[Avi_Stream_Pos])
                    Codec_External[Avi_Stream_Pos]=new File_Mpeg4v;

                //Analysing
                if (Codec_External[Avi_Stream_Pos]->Read(Begin+Offset, Element_Size[Level])>0)
                {
                    Fill(Stream_Visual, 0, "Encoded_Library", Codec_External[Avi_Stream_Pos]->Get(Stream_Visual, 0, _T("Encoded_Library")));
                    Fill(Stream_Visual, 0, "Codec_Settings", Codec_External[Avi_Stream_Pos]->Get(Stream_Visual, 0, _T("Codec_Settings")));
                    Fill(Stream_Visual, 0, "Resolution", Codec_External[Avi_Stream_Pos]->Get(Stream_Visual, 0, _T("Resolution")));
                    Fill(Stream_Visual, 0, "Interlacement", Codec_External[Avi_Stream_Pos]->Get(Stream_Visual, 0, _T("Interlacement")));
                    Fill(Stream_Visual, 0, "Chroma", Codec_External[Avi_Stream_Pos]->Get(Stream_Visual, 0, _T("Chroma")));
                }
            }
        }
        else if (Avi_Stream_Kind==_T("wb"))
        {
			size_t Audio_Pos=0; //TODO: a better handling of the position
            if (Avi_Stream_Pos>0)
                Audio_Pos=Avi_Stream_Pos-1;
            Ztring Codec=Get(Stream_Audio, 0, _T("Codec"));
            //MPEG-1 or 2
            if (Config.Codec_Get(Get(Stream_Audio, 0, _T("Codec")), InfoCodec_KindofCodec).find(_T("MPEG-1"))==0)
            {
                //We want to analyse only the first chunk
                if (!Codec_External[Avi_Stream_Pos])
                {
                    Codec_External[Avi_Stream_Pos]=new File_Mpega;

                    //Analysing
                    if (Codec_External[Avi_Stream_Pos]->Open(Begin+Offset, Element_Size[Level])>0)
                    {
                        Fill(Stream_Audio, Audio_Pos, "Codec/String", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Codec/String")));
                        Fill(Stream_Audio, Audio_Pos, "Codec_Profile", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Codec_Profile")));
                        Fill(Stream_Audio, Audio_Pos, "BitRate_Mode", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("BitRate_Mode")));
                        Fill(Stream_Audio, Audio_Pos, "Resolution", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Resolution")));
                        Fill(Stream_Audio, Audio_Pos, "Encoded_Library", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Encoded_Library")));
                    }
                }
            }
            //AC3
            else if (Codec==_T("2000"))
            {
                //We want to analyse only the first chunk
                if (!Codec_External[Avi_Stream_Pos])
                {
                    Codec_External[Avi_Stream_Pos]=new File_Ac3;

                    //Analysing
                    if (Codec_External[Avi_Stream_Pos]->Open(Begin+Offset, Element_Size[Level])>0)
                    {
                        Fill(Stream_Audio, Audio_Pos, "Codec/String", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Codec/String")));
                        Fill(Stream_Audio, Audio_Pos, "Codec_Profile", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Codec_Profile")));
                        Fill(Stream_Audio, Audio_Pos, "BitRate_Mode", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("BitRate_Mode")));
                        Fill(Stream_Audio, Audio_Pos, "Channel(s)", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Channel(s)")));
                        Fill(Stream_Audio, Audio_Pos, "ChannelPositions", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("ChannelPositions")));
                        Fill(Stream_Audio, Audio_Pos, "Resolution", Codec_External[Avi_Stream_Pos]->Get(Stream_Audio, 0, _T("Resolution")));
                    }
                }
            }
        }
        else if (Avi_Stream_Kind==_T("tx"))
        {
            // Text
            // "GAB2"               4 bytes
            // 0x00                 1 byte
            // 0x0002               2 bytes, this is Unicode
            // Name_Size (bytes)    4 bytes
            // Name (UTF-16)        Name_Size bytes
            // 0x04                 2 bytes
            // File_Size            4 bytes
            // File                 File_Size bytes, entire SRT/SS file
        }
    }
}

//---------------------------------------------------------------------------
void File_Riff::Strf_Analyse()
{
    //Coherancy test
    if (Strh_StreamType==Error)
        return;

    //Parse depending of kind of stream
         if (Strh_StreamType==CC4("vids")) Strf_Vids_Analyse();
    else if (Strh_StreamType==CC4("iavs")) Strf_Iavs_Analyse();
    else if (Strh_StreamType==CC4("auds")) Strf_Auds_Analyse();
    else if (Strh_StreamType==CC4("mids")) Strf_Mids_Analyse();
    else if (Strh_StreamType==CC4("txts")) Strf_Txts_Analyse();
}

//---------------------------------------------------------------------------
// Chunk "strf", at least 16 bytes for fccType "auds"
// FormatTag                        2 bytes, Pos=0
// Channels                         2 bytes, Pos=2
// SamplesPerSec                    4 bytes, Pos=4
// AvgBytesPerSec                   4 bytes, Pos=8
// BlockAlign                       2 bytes, Pos=12
// BitsPerSample                    2 bytes, Pos=14
// cbSize                           2 bytes, Pos=16
// Options
//
// Options could be:
// ValidBitsPerSample               2 bytes, Pos=18 (Optional)
// SamplesPerBlock                  2 bytes, Pos=20 (Optional)
// Reserved                         2 bytes, Pos=22 (Optional)
// ChannelMask                      4 bytes, Pos=24 (Optional)
// SubFormat                        16 bytes, Pos=28 (Optional)
//
// For FormatTag=0x0055
// ID                               2 bytes
// Flags                            4 bytes
// BlockSize                        2 bytes
// FramesPerBlock                   2 bytes
// CodecDelay                       2 bytes
//
void File_Riff::Strf_Auds_Analyse()
{
    //Coherancy test
    if (Element_Size[Level]<16) //Size of atom
        return;

    //Fill Audio datas
    Stream_Prepare(Stream_Audio);
    Fill("Codec", LittleEndian2int16u(Begin+Offset+0), 16); //FormatTag
    Fill("Channel(s)", LittleEndian2int16u(Begin+Offset+2)!=5?LittleEndian2int16u(Begin+Offset+2):6); //Channels, some workaround about AC3 (for example) with subwoofer
    Fill("SamplingRate", LittleEndian2int16u(Begin+Offset+4)); //SamplesPerSec
    Fill("BitRate", LittleEndian2int16u(Begin+Offset+8)*8); //AvgBytesPerSec
    if (LittleEndian2int16u(Begin+Offset+14))
        Fill("Resolution", LittleEndian2int16u(Begin+Offset+14)); //BitsPerSample

    //Option - MP3
    if (LittleEndian2int16u(Begin+Offset+0)==0x0055)
    {
    }
    //Option - AAC
    else if (LittleEndian2int16u(Begin+Offset+0)==0x0AAC)
    {
        if (LittleEndian2int16u(Begin+Offset+16)==5) //Stream with SBR has 5 bytes option
            Fill("Codec_Profile", "SBR");
    }
    //Options - Template
    else
    {
        if (Element_Size[Level]!=28 || LittleEndian2int16u(Begin+Offset+16)!=10) //No known optional fields
            return;
        Fill("Resolution", LittleEndian2int16u(Begin+Offset+18));
        int32u Position=LittleEndian2int32u(Begin+Offset+26);
        Ztring Front, Middle, Rear, Top, Surround, ChannelPositions;
        if (Position&0x00000001) Front   +=_T(" L");
        if (Position&0x00000004) Front   +=_T(" C");
        if (Position&0x00000040) Front   +=_T(" C"); //Center Left
        if (Position&0x00000080) Front   +=_T(" C"); //Center Rigth
        if (Position&0x00000002) Front   +=_T(" R");
        if (Position&0x00000200) Middle  +=_T(" L");
        if (Position&0x00000400) Middle  +=_T(" R");
        if (Position&0x00000010) Rear    +=_T(" L");
        if (Position&0x00000100) Rear    +=_T(" C");
        if (Position&0x00000020) Rear    +=_T(" R");
        if (Position&0x00000008) Surround+=_T(" C");
        if (Position&0x00000800) Top     +=_T(" C");
        if (Position&0x00000800) Top     +=_T(" FL");
        if (Position&0x00002000) Top     +=_T(" FC");
        if (Position&0x00001000) Top     +=_T(" FR");
        if (Position&0x00000800) Top     +=_T(" RL");
        if (Position&0x00010000) Top     +=_T(" RC");
        if (Position&0x00020000) Top     +=_T(" RR");
        if (!Front.empty()   ) {ChannelPositions+=_T("Front"); ChannelPositions+=_T(": "); ChannelPositions+=Front   ; ChannelPositions+=_T(", ");}
        if (!Middle.empty()  ) {ChannelPositions+=_T("Front"); ChannelPositions+=_T(": "); ChannelPositions+=Middle  ; ChannelPositions+=_T(", ");}
        if (!Rear.empty()    ) {ChannelPositions+=_T("Front"); ChannelPositions+=_T(": "); ChannelPositions+=Rear    ; ChannelPositions+=_T(", ");}
        if (!Surround.empty()) {ChannelPositions+=_T("Front"); ChannelPositions+=_T(": "); ChannelPositions+=Surround; ChannelPositions+=_T(", ");}
        if (!Top.empty()     ) {ChannelPositions+=_T("Front"); ChannelPositions+=_T(": "); ChannelPositions+=Top     ; ChannelPositions+=_T(", ");}
        ChannelPositions.resize(ChannelPositions.size()-2);
        Fill("ChannelPositions", ChannelPositions);
    }
}

//---------------------------------------------------------------------------
// Chunk "strf", 32 bytes for fccType "iavs"
// DVAAuxSrc                        4 bytes, Pos=0
// DVAAuxCtl                        4 bytes, Pos=4
// DVAAuxSrc1                       4 bytes, Pos=8
// DVAAuxCtl1                       4 bytes, Pos=12
// DVVAuxSrc                        4 bytes, Pos=16
// DVVAuxCtl                        4 bytes, Pos=20
// DVReserved                       8 bytes, Pos=24
//
void File_Riff::Strf_Iavs_Analyse()
{
    //Coherancy test
    if (Element_Size[Level]<32 || Strh_Offset==Error) //size of atom
        return;

    //Filling
    Stream_Prepare(Stream_Visual);
    Fill("Codec", (const char*)(Begin+Strh_Offset+4), 4); //Compression

    //FrameRate
    FrameRate=0;
    if (LittleEndian2int32u(Begin+Strh_Offset+20)!=0) //StreamHeader Scale
    {
        FrameRate=((float)LittleEndian2int32u(Begin+Strh_Offset+24))/LittleEndian2int32u(Begin+Strh_Offset+20); //StreamHeader Rate / StreamHeader Scale
        Fill("FrameRate", FrameRate, 3);
    }

    //Fill datas depend of the codec
    int32u Compression=CC4(Begin+Strh_Offset+4); //Compression
         if (Compression==CC4("dvsd") || Compression==CC4("dvsd"))
    {
                                    Fill("Width",  720);
             if (FrameRate==25.000) Fill("Height", 576);
        else if (FrameRate==29.970) Fill("Height", 480);
    }
    else if (Compression==CC4("dvhd"))
    {
                                    Fill("Width",  1440);
             if (FrameRate==25.000) Fill("Height", 1152);
        else if (FrameRate==30.000) Fill("Height",  960);
    }
}

//---------------------------------------------------------------------------
// Chunk "strf", unknown size for fccType "mids"
// Unknown format
//
void File_Riff::Strf_Mids_Analyse()
{
    Stream_Prepare(Stream_Audio);
    Fill("Codec", "Midi");
}

//---------------------------------------------------------------------------
// Chunk "strf", unknown size for fccType "txts"
// Unknown format
//
void File_Riff::Strf_Txts_Analyse()
{
    Stream_Prepare(Stream_Text);
    Fill("Codec", "SRT");
}

//---------------------------------------------------------------------------
// Chunk "strf", 40 bytes for fccType "vids"
// Size                             4 bytes, Pos=0
// Width                            4 bytes, Pos=4
// Height                           4 bytes, Pos=8
// Planes                           2 bytes, Pos=12
// BitCount                         2 bytes, Pos=14
// Compression                      4 bytes, Pos=16
// SizeImage                        4 bytes, Pos=20
// XPelsPerMeter                    4 bytes, Pos=24
// YPelsPerMeter                    4 bytes, Pos=28
// ClrUsed                          4 bytes, Pos=32
// ClrImportant                     4 bytes, Pos=36
void File_Riff::Strf_Vids_Analyse()
{
    //Coherancy test
    if (Element_Size[Level]<40 || Strh_Offset==Error) //Size of atom
        return;

    //Filling
    int32u Compression=CC4(Begin+Offset+16); //Compression
    if ((Compression&0x000000FF)>=0x00000020 && (Compression&0x000000FF)<=0x0000007E
     && (Compression&0x0000FF00)>=0x00002000 && (Compression&0x0000FF00)<=0x00007E00
     && (Compression&0x00FF0000)>=0x00200000 && (Compression&0x00FF0000)<=0x007E0000
     && (Compression&0xFF000000)>=0x20000000 && (Compression&0xFF000000)<=0x7E000000
       ) //Sometimes this value is wrong, we have to test this
    {
        //Divx.com hack for subtitle, this is a text stream in a DivX container
        if (Compression==CC4("DXSB"))
        {
            Fill(Stream_General, 0, "Format", "DivX");
            Stream_Prepare(Stream_Text);
        }
        else
            Stream_Prepare(Stream_Visual);

        //Filling
        Fill("Codec", (const char*)(Begin+Offset+16), 4); //Compression
        Fill("Width", LittleEndian2int32u(Begin+Offset+4)); //Width
        Fill("Height", LittleEndian2int32u(Begin+Offset+8)); //Height
    }
    else
    {
        //Some Stream headers are broken, must use AVISTREAMINFOA structure instead of AVIFILEINFOA
        Stream_Prepare(Stream_Visual);
        Fill("Codec", (const char*)(Begin+Strh_Offset+4), 4); //StreamHeader fcchandler
        Fill("Width", LittleEndian2int16u(Begin+Strh_Offset+52)); //StreamHeader Width
        Fill("Height", LittleEndian2int16u(Begin+Strh_Offset+54)); //StreamHeader Height
    }

    //FrameRate
    if (Compression!=CC4("DXSB")) //Don(t calculate the framerate of DXSB, this is taext, not video
    {
        FrameRate=0;
        if (LittleEndian2int32u(Begin+Strh_Offset+20)!=0) //StreamHeader Scale
        {
            FrameRate=((float)LittleEndian2int32u(Begin+Strh_Offset+24))/LittleEndian2int32u(Begin+Strh_Offset+20); //StreamHeader Rate / StreamHeader Scale
            Fill("FrameRate", FrameRate, 3);
        }
    }
}

//---------------------------------------------------------------------------
// Chunk "strh", 56 bytes
// fccType                          4 bytes, Pos=0
// fccHandler                       4 bytes, Pos=4
// Flags                            4 bytes, Pos=8
// Priority                         2 bytes, Pos=12
// Language                         2 bytes, Pos=14
// InitialFrames                    4 bytes, Pos=16
// Scale                            4 bytes, Pos=20
// Rate                             4 bytes, Pos=24 (Rate/Scale is stream tick rate in ticks/sec)
// Start                            4 bytes, Pos=28
// Length                           4 bytes, Pos=32
// SuggestedBufferSize              4 bytes, Pos=36
// Quality                          4 bytes, Pos=40
// SampleSize                       4 bytes, Pos=44
// Frame_Left                       2 bytes, Pos=48
// Frame_Top                        2 bytes, Pos=50
// Frame_Rigth                      2 bytes, Pos=52
// Frame_Bottom                     2 bytes, Pos=54
//
// Flags:
// AVISF_DISABLED                   0x???????? Stream should not be activated by default
// AVISF_VIDEO_PALCHANGES           0x???????? Stream is a video stream using palettes where the palette is changing during playback.
//
void File_Riff::Strh_Analyse()
{
    //Coherancy test
    if (Element_Size[Level]<56) //Size of atom
        return;

    //Save info for later use
    Strh_Offset=Offset;
    Strh_StreamType=CC4(Begin+Offset+0); //fccType
}

//---------------------------------------------------------------------------
// Chunk "strl", this is a list
void File_Riff::Strl_Analyse()
{
    //Reset internal datas
    Strh_Offset=Error;
    Strh_StreamType=Error;
}

//---------------------------------------------------------------------------
// Chunk "strn", user defined size
// StreamName                       X bytes, Pos=0
//
void File_Riff::Strn_Analyse()
{
    Fill("Title", (char*)(Begin+Offset+0), Element_Size[Level]); //StreamName
}

//---------------------------------------------------------------------------
// Chunk "vprp", at least 36 bytes
// VideoFormatToken                 4 bytes, Pos=0
// VideoStandard                    4 bytes, Pos=4
// VerticalRefreshRate              4 bytes, Pos=8
// HTotalInT                        4 bytes, Pos=12
// VTotalInLines                    4 bytes, Pos=16
// FrameAspectRatio                 4 bytes, Pos=20
// FrameWidthInPixels               4 bytes, Pos=24
// FrameHeightInLines               4 bytes, Pos=28
// FieldPerFrame                    4 bytes, Pos=32
// FieldInfo[1]                     32 bytes, Pos=36 (Optional)
// ...
// FieldInfo[FieldPerFrame]         32 bytes, Pos=36+32x(FieldPerFrame-1) (Optional)
//
// FieldInfo, 32 bytes:
// CompressedBMHeight               4 bytes, Pos=0
// CompressedBMWidth                4 bytes, Pos=4
// ValidBMHeight                    4 bytes, Pos=8
// ValidBMWidth                     4 bytes, Pos=12
// ValidBMXOffset                   4 bytes, Pos=16
// ValidBMYOffset                   4 bytes, Pos=20
// VideoXOffsetInT                  4 bytes, Pos=24
// VideoYValidStartLine             4 bytes, Pos=28
//
void File_Riff::Vprp_Analyse()
{
}

//---------------------------------------------------------------------------
void File_Riff::HowTo()
{
    Stream_Prepare(Stream_General);
    Fill_HowTo("Format", "R");
    Fill_HowTo("OveralBitRate", "R");
    Fill_HowTo("PlayTime", "R");
    Fill_HowTo("Movie", "R INAM");
    Fill_HowTo("Track", "N INAM");
    Fill_HowTo("Track/Position", "N IPRT");
    Fill_HowTo("Album/Track_Total", "N IFRM");
    Fill_HowTo("Composer", "R IMUS");
    Fill_HowTo("WrittenBy", "R IWRI");
    Fill_HowTo("Director", "R IART");
    Fill_HowTo("DirectorOfPhotography", "R ICNM");
    Fill_HowTo("EditedBy", "R IEDT");
    Fill_HowTo("Producer", "R IPRO");
    Fill_HowTo("ProductionDesigner", "R IPDS");
    Fill_HowTo("CostumeDesigner", "R ICDS");
    Fill_HowTo("MasteredBy", "R IENG");
    Fill_HowTo("ProductionStudio", "R ISTD");
    Fill_HowTo("DistributedBy", "R IDST");
    Fill_HowTo("EncodedBy", "R ITCH");
    Fill_HowTo("CommissionedBy", "R ICMS");
    Fill_HowTo("Encoded_Original/DistributedBy", "R ISRC");
    Fill_HowTo("Subject", "R ISBJ");
    Fill_HowTo("Keywords", "R IKEY");
    Fill_HowTo("LawRating", "R IRTD");
    Fill_HowTo("Language", "R ILNG");
    Fill_HowTo("Medium", "R IMED");
    Fill_HowTo("Product", "R IPRD");
    Fill_HowTo("Country", "R ICNT");
    Fill_HowTo("Written_Date", "R ICRD");
    Fill_HowTo("Mastered_Date", "R IDIT");
    Fill_HowTo("Archival_Location", "R IARL");
    Fill_HowTo("Genre", "R IGNR");
    Fill_HowTo("Comment", "R ICMT");
    Fill_HowTo("Encoded_Application", "R ISFT");
    Fill_HowTo("Encoded_Original", "R ISRF");
    Fill_HowTo("Copyright", "R ICOP");

    Stream_Prepare(Stream_Visual);
    Fill_HowTo("Codec", "R");
    Fill_HowTo("FrameRate", "R");
    Fill_HowTo("FrameCount", "R");
    Fill_HowTo("Width", "R");
    Fill_HowTo("Height", "R");
    Fill_HowTo("AspectRatio", "R");
    Fill_HowTo("BitRate", "R");

    Stream_Prepare(Stream_Audio);
    Fill_HowTo("BitRate", "R");
    Fill_HowTo("Channel(s)", "R");
    Fill_HowTo("SamplingRate", "R");
    Fill_HowTo("Codec", "R");
}

} //NameSpace

#endif //MEDIAINFO_RIFF_*









