// File_Qt - Info for QuickTime files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Information about Quicktime files
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_QT_YES) || (!(defined(MEDIAINFO_VIDEO_NO) && defined(MEDIAINFO_AUDIO_NO)) && !defined(MEDIAINFO_QT_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
    #pragma hdrstop
#endif
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#include <wx/file.h>
#include <ZenLib/Ztring.h>
#include <ZenLib/Utils.h>
#include <zlib/zlib.h>
#include "MediaInfo/Multiple/File_Qt.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
extern MediaInfo_Config Config;
//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

int File_Qt::Read(const int8u* Begin, size_t Begin_Size, const int8u* End, size_t End_Size, int64u FileSize)
{
    if (Begin_Size<128)
        return -1;

    //Why? Impossible with next test
    size_t Offset=0;
    if (CC4(Begin+4)==CC4("mdat"))
    {
        //Header is at the end
        Offset=BigEndian2int32u((char*)Begin+Offset);
        if (Offset<FileSize && FileSize-Offset<End_Size)
        {
            Begin=End;
            Begin_Size=End_Size;
            Offset=Offset+End_Size-FileSize;
        }
        else
        {
            Stream_Prepare(Stream_General);
            General[0](_T("Format"))=_T("Quicktime");
            General[0](_T("Format/String"))=_T("Quicktime");
            General[0](_T("Format/Extensions"))=_T("QT MOV");
            General[0](_T("Format/Url"))=_T("http://www.apple.com/quicktime/download/standalone.html");
            return 1;
        }
    }
    if (CC4(Begin+4)==CC4("ftyp4"))
    {
        return -1; //This is a MPEG-4 file, let the MPEG-4 parser do its job
        //Offset=BigEndian2int32u((char*)Begin);
        //if (Offset+128>=Begin_Size)
        //    return -3;
    }


        if (CC4(Begin+Offset+4)==CC4("free") || CC4(Begin+Offset+4)==CC4("moov"))
    {
        //most common Quicktime video file
    }
    else if (CC4(Begin+Offset+4)==CC4("wide") || CC4(Begin+Offset+4)==CC4("skip"))
    {
        //Header is at the end
        Offset=8+BigEndian2int32u((char*)Begin+Offset+8);
        if (Offset<FileSize && FileSize-Offset<End_Size)
        {
            Begin=End;
            Begin_Size=End_Size;
            Offset=Offset+End_Size-FileSize;
        }
        else
        {
            Stream_Prepare(Stream_General);
            General[0](_T("Format"))=_T("Quicktime");
            General[0](_T("Format/String"))=_T("Quicktime");
            General[0](_T("Format/Extensions"))=_T("QT MOV");
            General[0](_T("Format/Url"))=_T("http://www.apple.com/quicktime/download/standalone.html");
            return 1;
        }
    }
    else if (CC4(Begin+Offset+4)==CC4("pckg"))
    {
        Stream_Prepare(Stream_General);
        General[0](_T("Format"))=_T("Quicktime Archive");
        General[0](_T("Format/String"))=_T("Quicktime compressed archive");
        General[0](_T("Format/Extensions"))=_T("QT MOV");
        General[0](_T("Format/Url"))=_T("http://www.apple.com/quicktime/download/standalone.html");
        return 1;
    }
    else if (CC4(Begin+Offset+4)==CC4("idsc") || CC4(Begin+Offset+4)==CC4("idat"))
    {
        Stream_Prepare(Stream_General);
        General[0](_T("Format"))=_T("Quicktime Image");
        General[0](_T("Format/String"))=_T("Quicktime image");
        General[0](_T("Format/Extensions"))=_T("QT MOV");
        General[0](_T("Format/Url"))=_T("http://www.apple.com/quicktime/download/standalone.html");
        Stream_Prepare(Stream_Visual);
        Video[0](_T("Codec"))=_T("Quicktime");
        return 1;
    }
    else if (CC4(Begin+Offset+4)==CC4("pnot"))
    {
        Stream_Prepare(Stream_General);
        General[0](_T("Format"))=_T("Quicktime");
        General[0](_T("Format/String"))=_T("Quicktime");
        General[0](_T("Format/Extensions"))=_T("QT MOV");
        General[0](_T("Format/Url"))=_T("http://www.apple.com/quicktime/download/standalone.html");
        return 1;
    }
    else
        return -1;

    Stream_Prepare(Stream_General);
    General[0](_T("Format"))=_T("Quicktime");
    General[0](_T("Format/String"))=_T("Quicktime");
    General[0](_T("Format/Extensions"))=_T("QT MOV");
    General[0](_T("Format/Url"))=_T("http://www.apple.com/quicktime/download/standalone.html");

    //Compressed header
    const int8u* Begin2=Begin;
    size_t Begin2_Size=Begin_Size;
    if(CC4(Begin+Offset+24)==CC4("zlib"))
    {
        unsigned long SourceLen=BigEndian2int32u((char*)Begin+Offset+28)-12;
        unsigned long DestLen=BigEndian2int32u((char*)Begin+Offset+36);
        Begin2=new int8u[DestLen];
        if (uncompress((Bytef*)Begin2, &DestLen, (Bytef*)Begin+Offset+28+12, SourceLen)<0)
        {
            delete[] Begin2;
            return 2;
        }
        Begin2_Size=DestLen;
        Offset=0;
    }

    while(Offset<=Begin2_Size-32)
    {
        //TODO : a better search
        while (Offset<=Begin2_Size-32 && CC4(Begin2+Offset)!=CC4("stsd"))
            Offset++;
        if (Offset<=Begin2_Size-32)
        {
            Offset+=12;

            //Detect kind of stream
            Ztring Codec; Codec.From_Local((char*)Begin2+Offset+4, 4);
            const Ztring& KindOfCodec=Config.Codec_Get(Codec, InfoCodec_KindOfStream);

            //Wave?
            if (KindOfCodec==_T("A") || Begin2[Offset+56]=='w')
            {
                //Audio
                size_t Audio_Count=Stream_Prepare(Stream_Audio);
                if (CC2(Begin+Offset+4)!=CC2("ms"))
                    Audio[Audio_Count](_T("Codec")).From_Local((char*)Begin2+Offset+4, 4);
                else //Based of Microsoft 2cc
                    Audio[Audio_Count](_T("Codec")).From_Number(BigEndian2int16u((char*)Begin2+Offset+6), 16);
                Audio[Audio_Count](_T("Channel(s)")).From_Number(BigEndian2int16u((char*)Begin2+Offset+24));
                Audio[Audio_Count](_T("Resolution")).From_Number(BigEndian2int16u((char*)Begin2+Offset+26));
                Audio[Audio_Count](_T("SamplingRate")).From_Number(BigEndian2int16u((char*)Begin2+Offset+32));
                Audio_Count++;
            }
            else
            {
                //Video
                size_t Video_Count=Stream_Prepare(Stream_Visual);
                Video[Video_Count](_T("Codec")).From_Local((char*)Begin2+Offset+4, 4);
                Video[Video_Count](_T("Width")).From_Number(BigEndian2int16u((char*)Begin2+Offset+32));
                Video[Video_Count](_T("Height")).From_Number(BigEndian2int16u((char*)Begin2+Offset+34));
                //size_t Size=BigEndian2int8u((char*)Begin2+Offset+50);
                //Video[Video_Count](_T("Codec/String")).From_Local((char*)Begin2+Offset+51, Size);
                Video[Video_Count](_T("Resolution")).From_Number(BigEndian2int8u((char*)Begin2+Offset+83));
                Video_Count++;
            }
        }
    }

    Offset=0;
    size_t PlayTime=0;
    while(Offset<=Begin2_Size-4)
    {
        while (Offset<=Begin2_Size-4 && CC4(Begin+Offset)!=CC4("mdhd"))
            Offset++;
        if (Offset<=Begin2_Size-4)
        {
            Offset+=16;
            size_t TimeScale=BigEndian2int32u((char*)Begin2+Offset);
            size_t DurationScale=BigEndian2int32u((char*)Begin2+Offset+4);
            size_t PlayTimeTemp=(size_t)(((float)DurationScale)/TimeScale*1000);
            if (PlayTimeTemp>PlayTime)
                PlayTime=PlayTimeTemp; //Take the Maximum of playtimes
        }
    }

    if (PlayTime>0)
        General[0](_T("PlayTime")).From_Number(PlayTime);

    if (Begin!=Begin2) //Compressed
        delete[] Begin2;

    Offset=0;
    while(Offset<=End_Size-4)
    {
        while (Offset<=End_Size-4 && !(CC4(End+Offset)==CC4("udta") && End[Offset+8]==0xA9))
            Offset++;
        if (Offset<=End_Size-4)
        {
            Offset+=4;
            size_t Size;
            Ztring Setting;
            while(Offset<End_Size-16 && End[Offset+4]==0xA9)
            {
                Size=BigEndian2int32u((char*)End+Offset);
                if (Offset+Size<End_Size)
                {
                    Ztring Name, Value;
                         if (CC3(End+Offset+5)==CC3("aut"))
                        {Name=_T("Author"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("cmt"))
                        {Name=_T("Comment"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("cpy"))
                        {Name=_T("Copyright"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("day"))
                        {Name=_T("Encoded_Date"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("des"))
                        {Name=_T("Title/More"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("dir"))
                        {Name=_T("Director"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("dis"))
                        {Name=_T("TermsOfUse"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("nam"))
                        {Name=_T("Title"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("hst"))
                        {Name=_T("HostComputer"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("inf"))
                        {Name=_T("Information"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("key"))
                        {Name=_T("Keywords"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("mak"))
                        {Name=_T("Make"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("mod"))
                        {Name=_T("Model"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("fmt"))
                        {Name=_T("Encoded_Original"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("src"))
                        {Name=_T("Encoded_Original/DistributedBy"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("prf"))
                        {Name=_T("Performer"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("prd"))
                        {Name=_T("Producer"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("PRD"))
                        {Name=_T("Product"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("swr"))
                        {Name=_T("Encoded_Application"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("req"))
                        {Name=_T("SpecialPlaybackRequirements"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("wrn"))
                        {Name=_T("Warning"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("wrt"))
                        {Name=_T("ScreenplayBy"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("ed1"))
                        {Name=_T("Tagged_Date"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else if (CC3(End+Offset+5)==CC3("chp"))
                        {Name=_T("Chapter"); Value.From_Local((char*)End+Offset+12, Size-12);}
                    else
                    {
                        Name.From_Local((char*)End+Offset+5, 3);
                        Value.From_Local((char*)End+Offset+12, Size-12);
                    }
                    if (!Name.empty() && !Value.empty())
                    {
                        if (!General[0](Name).empty())
                        {
                            General[0](Name)+=_T(" / ");
                            General[0](Name)+=Value;
                        }
                        else
                            General[0](Name)=Value;
                    }
                    Offset+=Size;
                }
            }
        }
    }

    return 1;
}


//---------------------------------------------------------------------------
void File_Qt::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
        General[0](_T("Movie"), Info_HowTo)=_T("R nam");
        General[0](_T("Movie/More"), Info_HowTo)=_T("R des"); //Description
        General[0](_T("Track"), Info_HowTo)=_T("R nam");
        General[0](_T("Track/More"), Info_HowTo)=_T("R des"); //Description
        General[0](_T("Performer"), Info_HowTo)=_T("R aut|prf"); //Author and Performer
        General[0](_T("Author"), Info_HowTo)=_T("R aut");
        General[0](_T("Comment"), Info_HowTo)=_T("R cmt");
        General[0](_T("Copyright"), Info_HowTo)=_T("R cpy");
        General[0](_T("Encoded_Date"), Info_HowTo)=_T("R day");
        General[0](_T("Director"), Info_HowTo)=_T("R dir");
        General[0](_T("TermsOfUse"), Info_HowTo)=_T("R dis"); //Disclaimer
        General[0](_T("Keywords"), Info_HowTo)=_T("R key");
        General[0](_T("Encoded_Original"), Info_HowTo)=_T("R fmt"); //OriginalFormat
        General[0](_T("Encoded_Original/DistributedBy"), Info_HowTo)=_T("R src"); //OriginalSource
        General[0](_T("Producer"), Info_HowTo)=_T("R prd");
        General[0](_T("Encoded_Application"), Info_HowTo)=_T("R swr"); //Software
        General[0](_T("ScreenplayBy"), Info_HowTo)=_T("R wrt"); //Writer
        General[0](_T("Tagged_Date"), Info_HowTo)=_T("R ed1"); //EditDate1
        //More tags are "translated" :
        //hst --> HostComputer
        //inf --> Information
        //mak --> Make
        //mod --> Model
        //PRD --> Product
        //req --> SpecialPlaybackRequirements
        //wrn --> Warning
        //chp --> Chapter
    }
    else if (StreamKind==Stream_Visual)
    {
        Video[0](_T("Codec"), Info_HowTo)=_T("R");
        Video[0](_T("BitRate"), Info_HowTo)=_T("R");
        Video[0](_T("Width"), Info_HowTo)=_T("R");
        Video[0](_T("Height"), Info_HowTo)=_T("R");
        Video[0](_T("AspectRatio"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
    }
}

} //NameSpace

#endif //MEDIAINFO_QT_*

