// File_Mpeg - Info for MPEG files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_MPEG_YES) || (!(defined(MEDIAINFO_VIDEO_NO) && defined(MEDIAINFO_AUDIO_NO)) && !defined(MEDIAINFO_MPEG_NO))
//---------------------------------------------------------------------------

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// MPEG Program Stream 1&2 - Codes
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Subtitle                         0x00000120 - 0x0000013F (Used?)
// AC3 or DTS                       0x00000180 - 0x0000018F (Used?)
// End                              0x000001B9
// Start                            0x000001BA
// System                           0x000001BB
// Program map                      0x000001BC
// Private                          0x000001BD              (Used for AC3 and DTS in DVD)
// Padding                          0x000001BE
// Private                          0x000001BF              (Used for AC3 and DTS in DVD)
// MPEG Audio                       0x000001C0 - 0x000001DF
// MPEG Video                       0x000001E0 - 0x000001EF
// ECM?                             0x000001F0
// EMM?                             0x000001F1
// DSMCC?                           0x000001F2
// ISO 13522?                       0x000001F3
// ITUT Type A to E?                0x000001F4 - 0x000001F8
// PS Directory                     0x000001FF
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// MPEG Program Stream 1&2 - PES
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Start                            3 bytes - 0x000001
// Code                             1 byte
// Size                             2 bytes (ONLY if code is different of 0xB9 or 0xBA)
// Datas                            x
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
    #pragma hdrstop
#endif
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#include <ZenLib/Utils.h>
#include "MediaInfo/Multiple/File_Mpeg.h"
#include "MediaInfo/Video/File_Mpegv.h"
#include "MediaInfo/Audio/File_Mpega.h"
#include "MediaInfo/Audio/File_Dts.h"
#include "MediaInfo/Audio/File_Ac3.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
int32u MPEG_InitTime=Error;
int32u MPEG_InitTime_Frame=0; //Because framerate is not know before InitTime
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
int64u MPEG_PTS_Read (const int8u* Begin);
//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

//---------------------------------------------------------------------------
int File_Mpeg::Read(const int8u* Begin_, size_t Begin_Size_, const int8u* End_, size_t End_Size_, int64u FileSize)
{
    Begin=Begin_;
    Begin_Size=Begin_Size_;
    End=End_;
    End_Size=End_Size_;
    Offset=0;
    MPEG_Version=0;

    //Integrity
    if (Begin_Size<16)
        return -1;
    if (CC4(Begin)==CC4("RIFF"))
        return -1; //Often in DAT files, and the parser is not enough presice to detect them

    //Loog for first Sync word
    while (Offset+128<Begin_Size && CC3(Begin+Offset)!=0x000001)
        Offset++;
    if (Offset+128>=Begin_Size)
        return -2;

    //Parse Begin with PES parser
    while (Offset+16<Begin_Size)
        if (MPEG_PES_Header()==Error)
            Offset=Begin_Size;

    //Coherancy tests
    if (General.empty() || (Video.empty() && Audio.empty()))
        return -1;

    return 1;
}

//---------------------------------------------------------------------------
size_t File_Mpeg::MPEG_PES_Header()
{
    //Keep out unuseful 0x00 in the stream
    while (Offset+32<Begin_Size && Begin[Offset+2]==0x00)
        Offset++;

    //Look for PES_Start
    if (Offset+32>=Begin_Size || CC3(Begin+Offset)!=0x000001)
    {
        //Stream error. Try to find again the PS Pack start code to resynchronize
        while (Offset+32<Begin_Size && CC3(Begin+Offset)!=0x000001 && CC1(Begin+Offset+3)!=0xBA)
            Offset++;
        if (Offset+32>=Begin_Size)
            return Error; //No success to find again the PS Pack start code
    }

   //PS Pack Start Code
         if (Begin[Offset+3]==0xBA)
    {
        if ((Begin[Offset+4]&0xC0)!=0x40) //MPEG-1
        {
            if (General.empty())
            {
                Stream_Prepare(Stream_General);
                General[0](_T("Format"))=_T("MPEG-1");
                General[0](_T("Format/String"))=_T("MPEG-1 PS");
                General[0](_T("Format/Extensions"))=_T("M1S M1P MPEG MPG MPE MPGX MPM");
                MPEG_Version=1;
            }
            Offset+=12;
        }
        else //MPEG-2
        {
            if (General.empty())
            {
                Stream_Prepare(Stream_General);
                General[0](_T("Format"))=_T("MPEG-2");
                General[0](_T("Format/String"))=_T("MPEG-2 PS");
                General[0](_T("Format/Extensions"))=_T("VOB M2S M2P MPEG MPG MPE MPGX MPM DAT");
                MPEG_Version=2;
            }
            int8u Padding=Begin[Offset+13]&0x07;
            Offset+=14+Padding;
        }
    }
    //PS End Code
    else if (Begin[Offset+3]==0xB9)
    {
        //Only skip it
        Offset+=2;
    }
    //Other Stream ID
    else
    {
        Stream_ID=Begin[Offset+3];
        int16u Size=BigEndian2int16u((char*)Begin+Offset+4); //PES size
        Offset_End=Offset+6+Size;
        if (Offset_End>=Begin_Size)
        {
            Offset=Offset_End;
            return 2;
        }
        Offset+=6;
        MPEG_Stream();
        Offset=Offset_End;
    }

    //Check coherency of video BitRate
    if (Video.size()==1 && General[0](_T("PlayTime")).To_int32s()!=0)
    {
        //Estimited video BitRate
        int32s OveralBitRate_Estimated=(int32s)(General[0](_T("FileSize")).To_float32()*8/General[0](_T("PlayTime")).To_float32()*1000.0);
        for (size_t I1=0; I1<Audio.size(); I1++)
            OveralBitRate_Estimated-=Audio[I1](_T("BitRate")).To_int32s();
        int32s OveralBitRate_Theoric=Video[0](_T("BitRate")).To_int32s();
        if (OveralBitRate_Theoric-OveralBitRate_Estimated>100000) // Difference >100 Kbps (overhead estimation)
        {
            //Video BitRate is false!
            Video[0](_T("BitRate")).From_Number((float32_int32s(((float)OveralBitRate_Estimated)/400)-125)*400); // 50 Kbps for Overhead
            Video[0](_T("BitRate_Mode"))=_T("VBR");
        }
    }

    return 1;
}

//---------------------------------------------------------------------------
size_t File_Mpeg::MPEG_PES_Data ()
{
    //Init
    TimeStamp.Have_PTS=false;
    TimeStamp.Have_DTS=false;

    //Keep out firsts 0xFF
    while (Offset<Offset_End && Begin[Offset]==0xFF)
        Offset++;
    if (Offset+16>=Offset_End)
        return 2;

    //Buffer scale and size
    if ((Begin[Offset]&0xC0)==0x40)
        Offset+=2;

    //MPEG-1 with PTS
    if ((Begin[Offset]&0xF0)==0x20)
    {
        TimeStamp.Have_PTS=true;
        TimeStamp.PTS=TimeStamp.DTS=MPEG_PTS_Read(Begin+Offset);
        Offset+=5;
    }

    //MPEG-1 with PTS and DTS
    else if ((Begin[Offset]&0xF0)==0x30)
    {
        TimeStamp.Have_PTS=true;
        TimeStamp.Have_DTS=true;
        TimeStamp.PTS=MPEG_PTS_Read(Begin+Offset);
        TimeStamp.DTS=MPEG_PTS_Read(Begin+Offset+5);
        Offset+=10;
    }

    //MPEG-2 PES header
    else if ((Begin[Offset]&0xC0)==0x80)
    {
        //0x80 Flags Size
        int8u Header_Flags=Begin[Offset+1];
        int8u Header_Size=Begin[Offset+2];
        Offset+=3;
        //MPEG-2 with PTS
             if ((Header_Flags&0xC0)==0x80)
        {
            TimeStamp.Have_PTS=true;
            TimeStamp.PTS=TimeStamp.DTS=MPEG_PTS_Read(Begin+Offset);
        }
        //MPEG-2 with PTS and DTS
        else if ((Header_Flags&0xC0)==0xC0)
        {
            TimeStamp.Have_PTS=true;
            TimeStamp.Have_DTS=true;
            TimeStamp.PTS=MPEG_PTS_Read(Begin+Offset);
            TimeStamp.DTS=MPEG_PTS_Read(Begin+Offset+5);
        }
        Offset+=Header_Size;
    }
    return 1;
}

//---------------------------------------------------------------------------
size_t File_Mpeg::MPEG_Stream()
{
    //Video
         if (Stream_ID>=0xE0 && Stream_ID<=0xEF)
    {
        size_t Pos=0;
        while (Pos<Stream_List.size() && Stream_List[Pos]!=Stream_ID)
            Pos++;
        if (Pos!=Stream_List.size())
            return 1; //Already parsed
        MPEG_PES_Data();

        File_Mpegv MI;
        MI.MPEG_Version=MPEG_Version;

        size_t End_Offset=0;
        size_t End_Offset_End=0;
        //Trying to look for the last video frame with a TimeStamp
        if (End_Size>128)
        {
            End_Offset=End_Size-128;
            while (End_Offset>0 && !(CC3(End+End_Offset)==0x000001 && End[End_Offset+3]==0xB8))
                End_Offset--;
            while (End_Offset>0 && !(CC3(End+End_Offset)==0x000001 && End[End_Offset+3]==Stream_ID))
                End_Offset--;
            End_Offset_End=End_Offset;
            if (End_Offset>0)
            {
                size_t Size=BigEndian2int16u((char*)End+End_Offset+4); //PES size
                if (End_Offset_End+Size<End_Size)
                    End_Offset_End+=Size;
                else
                    End_Offset_End=End_Size;
            }
        }
        if (MI.Read(Begin+Offset, Offset_End-Offset, End+End_Offset, End_Offset_End-End_Offset)>0)
        {
            size_t Video_Count=Stream_Prepare(Stream_Visual);
            if (MI.Video.size()>0)
                Video[Video_Count]=MI.Video[0];
            if (TimeStamp.Have_PTS)
                Video[Video_Count](_T("Delay")).From_Number(TimeStamp.PTS);
            Stream_List.push_back(Stream_ID);

            //Tags in MPEG Video
            if (!General.empty() && !MI.General.empty() && !MI.General[0](_T("Encoded_Application")).empty())
                General[0](_T("Encoded_Application"))=MI.General[0](_T("Encoded_Application"));
        }
    }

    //Audio
    else if (Stream_ID>=0xC0 && Stream_ID<=0xDF)
    {
        size_t Pos=0;
        while (Pos<Stream_List.size() && Stream_List[Pos]!=Stream_ID)
            Pos++;
        if (Pos!=Stream_List.size())
            return 1; //Already parsed
        MPEG_PES_Data();

        File_Mpega MI;
        if (MI.Read(Begin+Offset, Offset_End-Offset)>0)
        {
            size_t Audio_Count=Stream_Prepare(Stream_Audio);
            if (MI.Audio.size()>0)
                Audio[Audio_Count]=MI.Audio[0];
            if (TimeStamp.Have_PTS)
                Audio[Audio_Count](_T("Delay")).From_Number(TimeStamp.PTS);
            Stream_List.push_back(Stream_ID);
        }
    }

    //Private
    else if (Stream_ID==0xBD || Stream_ID==0xBF)
    {
        //Private : often used in DVD for Subtitle, PCM, AC3 and DTS
        int8u Private_ID=Begin[Offset+1];
        size_t Pos=0;
        while (Pos<Private_List.size() && Private_List[Pos]!=Private_ID)
            Pos++;
        if (Pos!=Private_List.size())
            return 1; //Already parsed
        MPEG_PES_Data();

        //Subtitles
             if (Private_ID>=0x20 && Private_ID<=0x3F)
        {
                size_t Text_Count=Stream_Prepare(Stream_Text);
                Text[Text_Count](_T("Codec"))=_T("2-bit RLE");
                if (TimeStamp.Have_PTS)
                    Text[Text_Count](_T("Delay")).From_Number(TimeStamp.PTS);
                Private_List.push_back(Private_ID);
        }

        //AC3
        else if (Private_ID>=0x80 && Private_ID<=0x88)
        {
            File_Ac3 MI;
            if (MI.Read(Begin+Offset, Offset_End-Offset)>0)
            {
                size_t Audio_Count=Stream_Prepare(Stream_Audio);
                if (MI.Audio.size()>0)
                    Audio[Audio_Count]=MI.Audio[0];
                if (TimeStamp.Have_PTS)
                    Audio[Audio_Count](_T("Delay")).From_Number(TimeStamp.PTS);
                Private_List.push_back(Private_ID);
            }
        }

        //DTS
        else if (Private_ID>=0x88 && Private_ID<=0x8F)
        {
            File_Dts MI;
            if (MI.Read(Begin+Offset, Offset_End-Offset)>0)
            {
                size_t Audio_Count=Stream_Prepare(Stream_Audio);
                if (MI.Audio.size()>0)
                    Audio[Audio_Count]=MI.Audio[0];
                if (TimeStamp.Have_PTS)
                    Audio[Audio_Count](_T("Delay")).From_Number(TimeStamp.PTS);
                Private_List.push_back(Private_ID);
            }
        }

        //PCM
        else if (Private_ID>=0xA0 && Private_ID<=0xA8)
        {
                size_t Audio_Count=Stream_Prepare(Stream_Audio);
                Audio[Audio_Count](_T("Codec"))=_T("PCM");
                //Resolution
                int Resolution=Begin[Offset+5]>>6;
                if (Resolution==0) Audio[Audio_Count](_T("Resolution"))=_T("16");
                if (Resolution==1) Audio[Audio_Count](_T("Resolution"))=_T("20");
                if (Resolution==2) Audio[Audio_Count](_T("Resolution"))=_T("24");
                int SamplingRate=(Begin[Offset+5]&0x30)>>4;
                if (SamplingRate==0) Audio[Audio_Count](_T("SamplingRate"))=_T("48000");
                if (SamplingRate==1) Audio[Audio_Count](_T("SamplingRate"))=_T("96000");
                int Channels=Begin[Offset+5]&0x07;
                Audio[Audio_Count](_T("Channel(s)")).From_Number(Channels+1);
                if (TimeStamp.Have_PTS)
                    Audio[Audio_Count](_T("Delay")).From_Number(TimeStamp.PTS);
                Private_List.push_back(Private_ID);
        }
    }

    return 1;
}

//---------------------------------------------------------------------------
int64u MPEG_PTS_Read (const int8u* Begin)
{
    int64u PTS;
    int16u I1;

    PTS=((Begin[0]>>1)&0x07);
    PTS<<=15;
    I1=BigEndian2int16u((char*)Begin+1)>>1;
    PTS|=I1;
    PTS<<=15;
    I1=BigEndian2int16u((char*)Begin+3)>>1;
    PTS|=I1;
    return PTS;
}


//---------------------------------------------------------------------------
void File_Mpeg::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
        General[0](_T("Encoded_Application"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Visual)
    {
        Video[0](_T("Codec"), Info_HowTo)=_T("R");
        Video[0](_T("BitRate"), Info_HowTo)=_T("R");
        Video[0](_T("Width"), Info_HowTo)=_T("R");
        Video[0](_T("Height"), Info_HowTo)=_T("R");
        Video[0](_T("AspectRatio"), Info_HowTo)=_T("R");
        Video[0](_T("FrameRate"), Info_HowTo)=_T("R");
        Video[0](_T("Delay"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Resolution"), Info_HowTo)=_T("R");
        Audio[0](_T("Delay"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Text)
    {
        Text[0](_T("Codec"), Info_HowTo)=_T("R");
        Text[0](_T("Delay"), Info_HowTo)=_T("R");
    }
}

} //NameSpace

#endif //MEDIAINFO_MPEG_*

