// File__Base - Base for other files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Give common methods for all file types
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
#ifndef MediaInfo__BaseH
#define MediaInfo__BaseH
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/MediaInfo_Config.h"
#include <ZenLib/ZtringListList.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//***************************************************************************
// Class File__Base
//***************************************************************************

class File__Base
{
public :
    //Gestion de la classe
    File__Base();
    virtual ~File__Base()                                                       {};

    //Analyze
    int     Open (const Ztring &CompleteFileName); //1 if succeed, 2 if problem with format, -1 if wrong type, -2 if no file
    int     Open (const int8u* Begin, size_t Begin_Size, const int8u* End=NULL, size_t End_Size=0, int64u FileSize=0); //1 if succeed, 2 if problem with format, -1 if wrong type
    int     Save ();

    //Get information
    Ztring  Inform ();
    Ztring  Inform (stream_t StreamKind, size_t StreamNumber=0); //All about only a specific stream
    const Ztring &Get (stream_t StreamKind, size_t StreamNumber, size_t Parameter, info_t KindOfInfo=Info_Text);
    const Ztring &Get (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, info_t KindOfInfo=Info_Text, info_t KindOfSearch=Info_Name);

    //Set information
    int           Set (stream_t StreamKind, size_t StreamNumber, size_t Parameter, const Ztring &ToSet, const Ztring &OldValue=_T(""));
    int           Set (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &ToSet, const Ztring &OldValue=_T(""));

    //Options
    size_t Count_Get (stream_t StreamKind, size_t Pos=Error) const;
    void   Language_Set (); //Update language for an existing File__Base

    //Infos
    ZtringListList Info_Capacities ();

protected :
    //Theses classes need access to internal structure for optimization. There is recursivity with theses formats
    friend class File_Riff;

    //Read
    virtual int Read        ();
    virtual int Read        (const int8u* Begin, size_t Begin_Size, const int8u* End=NULL, size_t End_Size=0, int64u FileSize=0); //To overload

    //Write
    virtual int Write       (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &ToSet, const Ztring &OldValue); //Write the value in memory
    virtual int WriteToDisk (); //Write modified tags on disk

    //How to
    virtual void HowTo (stream_t StreamKind);

    //Before filling the stream, the stream must be prepared
    size_t Stream_Prepare   (stream_t KindOfStream);
    void   General_Fill     (); //Special : pre-fill General with some important information

    //Fill with datas
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const Ztring  &Value);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const char*    Value, intu Value_Size=Error, bool Utf8=true);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const wchar_t* Value, intu Value_Size=Error);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int16u         Value, intu Radix=10);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int32u         Value, intu Radix=10);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int32s         Value, intu Radix=10);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int64u         Value, intu Radix=10);
    void Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, float32        Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing);
    void Fill_HowTo (stream_t StreamKind, size_t StreamPos, const char* Parameter, const char* Value);
    inline void Fill (const char* Parameter, const Ztring  &Value, intu ValueSize=Error, bool Utf8=true) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value);} //With the last set
    inline void Fill (const char* Parameter, const char*    Value, intu ValueSize=Error, bool Utf8=true) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, ValueSize, Utf8);} //With the last set
    inline void Fill (const char* Parameter, const int8u*   Value, intu ValueSize=Error, bool Utf8=true) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, (const char*)Value, ValueSize, Utf8);} //With the last set
    inline void Fill (const char* Parameter, const wchar_t* Value, intu ValueSize=Error) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, ValueSize);} //With the last set
    inline void Fill (const char* Parameter, int16u         Value, intu Radix=10) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int32u         Value, intu Radix=10) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int32s         Value, intu Radix=10) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, int64u         Value, intu Radix=10) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, Radix);} //With the last set
    inline void Fill (const char* Parameter, float32        Value, intu AfterComma=3, ztring_t Options=Ztring_Nothing) {Fill(KindOfStream_Last, StreamPos_Last, Parameter, Value, AfterComma, Options);} //With the last set
    inline void Fill_HowTo (const char* Parameter, const char* Value) {Fill_HowTo(KindOfStream_Last, StreamPos_Last, Parameter, Value);} //With the last set

    //Arrays
    std::vector<ZtringListList> General;
    std::vector<ZtringListList> Video;
    std::vector<ZtringListList> Audio;
    std::vector<ZtringListList> Text;
    std::vector<ZtringListList> Chapters;
    std::vector<ZtringListList>* Stream[Stream_Max]; //pointer to others listed streams

    //Optimization
    Ztring            CompleteFileName; //Cached filename for Read()
    std::vector<bool> Optimized[Stream_Max]; //If true, Arrays are not fully filled
    stream_t KindOfStream_Last;
    size_t   StreamPos_Last;

    //Utils
    void Merge(const File__Base &ToAdd); //Merge 2 File_Base

protected :
    //Divers
    void Clear();

private :
    void Finalize();

public :  //A virer
    void Traiter(Ztring &C); //enleve les $if...
};

} //NameSpace

#endif
