// File__Base - Base for other files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//
// Give common methods for all file types
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
#include "MediaInfo/File__Base.h"
#include <ZenLib/Trace.h>
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
extern MediaInfo_Config Config;
//---------------------------------------------------------------------------

//***************************************************************************
// Gestion de la classe
//***************************************************************************

//---------------------------------------------------------------------------
//Constructeurs
File__Base::File__Base ()
{
    //Init pointers
    Stream[Stream_General]=&General;
    Stream[Stream_Visual]=&Video;
    Stream[Stream_Audio]=&Audio;
    Stream[Stream_Text]=&Text;
    Stream[Stream_Chapters]=&Chapters;
}

//***************************************************************************
// Fonctions
//***************************************************************************

//---------------------------------------------------------------------------
int File__Base::Open (const Ztring &Fichier)
{
    Clear();
    CompleteFileName=Fichier;
    int Resultat=Read();
    if (Resultat>0)
        Finalize();
    return Resultat;
}

int File__Base::Open (const int8u* Begin, size_t Begin_Size, const int8u* End, size_t End_Size, int64u FileSize)
{
    Clear();
    int Resultat=Read(Begin, Begin_Size, End, End_Size, FileSize);
    if (Resultat>0)
        Finalize();
    return Resultat;
}

//---------------------------------------------------------------------------
size_t File__Base::Count_Get (stream_t StreamKind, size_t Pos) const
{
    //Integrity
    if (StreamKind>=Stream_Max)
        return Error;

    //Count of streams
    if (Pos==Error)
        return (*Stream[StreamKind]).size();

    //Integrity
    if (Pos>=(*Stream[StreamKind]).size())
        return Error;

    //Count of piece of information in a stream
    return (*Stream[StreamKind])[Pos].size();
}

//---------------------------------------------------------------------------
const Ztring &File__Base::Get (stream_t StreamKind, size_t StreamNumber, size_t Parameter, info_t KindOfInfo)
{
    //Check integrity
    if (StreamKind>=Stream_Max || StreamNumber>=(*Stream[StreamKind]).size() || Parameter>=(*Stream[StreamKind])[StreamNumber].size() || KindOfInfo>=Info_Max)
        return Config.EmptyString_Get(); //Parameter is unknown

    //OK for Optimization?
    if (Optimized[StreamKind][StreamNumber] && ((*Stream[StreamKind])[StreamNumber].size()-Config.Info_Get(StreamKind).size()!=0 || KindOfInfo==Info_HowTo))
    {
        //Can't be optimized
        //-HowTo
        if (StreamNumber==0)
            HowTo(StreamKind);
        //-Others
        for (size_t Pos=0; Pos<(*Stream[StreamKind])[StreamNumber].size(); Pos++)
        {
            size_t Pos_Info=Config.Info_Get(StreamKind).Find((*Stream[StreamKind])[StreamNumber][Pos][0]);
            if (Pos_Info!=Error)
                for (size_t Pos1=Config.Info_Get(StreamKind)[Pos_Info].size()-1; Pos1>=Info_Measure; Pos1--) //-- for optimization of ZtringList
                     (*Stream[StreamKind])[StreamNumber][Pos](Pos1)=Config.Info_Get(StreamKind)[Pos_Info][Pos1];
        }
        Optimized[StreamKind][StreamNumber]=false;
    }

    //Verify validity of strings
    //-Info_Options
     if (!Optimized[StreamKind][StreamNumber] && KindOfInfo==Info_Options && (*Stream[StreamKind])[StreamNumber](Parameter, Info_Options).empty())
        (*Stream[StreamKind])[StreamNumber](Parameter, Info_Options)=_T("Y YT");

    //Optimization : KindOfInfo>Info_Text is in static lists
    if (Optimized[StreamKind][StreamNumber] && KindOfInfo>=Info_Measure && KindOfInfo<Config.Info_Get(StreamKind)[Parameter].size())
        return Config.Info_Get(StreamKind)[Parameter][KindOfInfo]; //look for static information only
    else
        return (*Stream[StreamKind])[StreamNumber][Parameter](KindOfInfo);
}

//---------------------------------------------------------------------------
const Ztring &File__Base::Get (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, info_t KindOfInfo, info_t KindOfSearch)
{
    TRACE(Trace+=_T("Get(B), StreamKind=");Trace+=ZenLib::Ztring::ToZtring((int8u)StreamKind);Trace+=_T(", StreamNumber=");Trace+=ZenLib::Ztring::ToZtring((int8u)StreamNumber);Trace+=_T(", Parameter=");Trace+=ZenLib::Ztring(Parameter);Trace+=_T(", KindOfInfo=");Trace+=ZenLib::Ztring::ToZtring((int8u)KindOfInfo);Trace+=_T(", KindOfSearch=");Trace+=ZenLib::Ztring::ToZtring((int8u)KindOfSearch);)
    //TRACE(Trace+=_T("Stream_Kind=");Trace+=ZenLib::Ztring::ToZtring((int8u)StreamKind);)
    //TRACE(Trace+=_T("Stream_Size=");Trace+=ZenLib::Ztring::ToZtring((int)(*Stream[StreamKind]).size());)

	intu ParameterI;

    //Check integrity
    if (StreamKind>=Stream_Max || StreamNumber>=(*Stream[StreamKind]).size() || (ParameterI=(*Stream[StreamKind])[StreamNumber].Find(Parameter, KindOfSearch))==Error || KindOfInfo>=Info_Max)
	{
		if (!(StreamKind>=Stream_Max || StreamNumber>=(*Stream[StreamKind]).size()))
		{
			TRACE(Trace+=_T("Get(B) error: General size=");Trace+=Ztring::ToZtring((int)(*Stream[StreamKind])[StreamNumber].Find(Parameter, KindOfSearch));)
		}
		return Config.EmptyString_Get(); //Parameter is unknown
	}
    //Legacy
    if (Parameter.find(_T("_String"))!=Error)
    {
        Ztring S1=Parameter;
        S1.FindAndReplace(_T("_String"), _T("/String"));
        return Get(StreamKind, StreamNumber, S1, KindOfInfo, KindOfSearch);
    }
    if (Parameter==_T("Channels"))
        return Get(StreamKind, StreamNumber, _T("Channel(s)"), KindOfInfo, KindOfSearch);

    //Special cases
    //-Inform for a stream
    if (Parameter==_T("Inform"))
        (*Stream[StreamKind])[StreamNumber](_T("Inform"))=Inform(StreamKind, StreamNumber);

	TRACE(Trace+=_T("Get(B), will return ");Trace+=Get(StreamKind, StreamNumber, ParameterI, KindOfInfo))

	return Get(StreamKind, StreamNumber, ParameterI, KindOfInfo);
}

//---------------------------------------------------------------------------
int File__Base::Set (stream_t StreamKind, size_t StreamNumber, size_t Parameter, const Ztring &ToSet, const Ztring &OldValue)
{
    return Set(StreamKind, StreamNumber, Get(StreamKind, StreamNumber, Parameter, Info_Name), ToSet, OldValue);
}

//---------------------------------------------------------------------------
int File__Base::Set (stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &ToSet, const Ztring &OldValue)
{
    return Write(StreamKind, StreamNumber, Parameter, ToSet, OldValue);
}

//---------------------------------------------------------------------------
ZtringListList File__Base::Info_Capacities()
{
    ZtringListList Retour;
    int Retour_Pos=0;
    for (size_t StreamKind=0; StreamKind<Stream_Max; StreamKind++)
    {
        Stream_Prepare((stream_t)StreamKind);
        Retour(Retour_Pos, 0)=Get((stream_t)StreamKind, 0, _T("StreamKind"), Info_Measure);
        Retour_Pos++;
        for (size_t Pos=0; Pos<Count_Get((stream_t)StreamKind, 0); Pos++)
        {
            if (Get((stream_t)StreamKind, 0, Pos, Info_Options)[InfoOption_ShowInSupported]==_T('Y'))
            {
                Retour(Retour_Pos, 0)=Get((stream_t)StreamKind, 0, Pos, Info_Name_Text);
                Retour(Retour_Pos, 1).assign(Get((stream_t)StreamKind, 0, Pos, Info_HowTo), 0, 1); //1 character only
                Retour_Pos++;
            }
        }
        Retour_Pos++;
    }

    return Retour;
}

//---------------------------------------------------------------------------
void File__Base::Language_Set()
{
/*
    for (size_t StreamKind=(size_t)Stream_General; StreamKind<(size_t)Stream_Max; StreamKind++)//Note : Optimisation, only the first Stream is, so StreamNumber is only 0
        for (size_t Pos=0; Pos<Config.Info[StreamKind].size(); Pos++)
        {
             //Info_Name_Text
             const Ztring &Z1=Config.Language_Get(Config.Info_Get((stream_t)StreamKind, Pos, Info_Name));
             if (Z1.empty())
                Set((stream_t)StreamKind, 0, Pos, Info_Name_Text, Config.Info_Get((stream_t) StreamKind, Pos, Info_Name));
             else
                Set((stream_t)StreamKind, 0, Pos, Info_Name_Text, Z1);
             //Info_Measure_Text
             const Ztring Z2=Config.Language_Get(Config.Info_Get((stream_t)StreamKind, Pos, Info_Measure));
             if (Z2.empty())
                Set((stream_t)StreamKind, 0, Pos, Info_Measure_Text, Config.Info_Get((stream_t)StreamKind, Pos, Info_Measure);
             else
                Set((stream_t)StreamKind, 0, Pos, Info_Measure_Text, Z2);
        }
*/
}

//***************************************************************************
// Preparation des streams
//***************************************************************************

//---------------------------------------------------------------------------
size_t File__Base::Stream_Prepare (stream_t KindOfStream)
{
    //Integrity
    if (KindOfStream>=Stream_Max)
        return Error;

    //Add a stream
    Stream[KindOfStream]->resize(Stream[KindOfStream]->size()+1);
    Optimized[KindOfStream].push_back(true);

    //Fill the stream
    ZtringListList* ZLL=&(*Stream[KindOfStream])[Stream[KindOfStream]->size()-1];
    ZLL->reserve(Config.Info_Get(KindOfStream).size());
    for (size_t Pos=0; Pos<Config.Info_Get(KindOfStream).size(); Pos++)
        (*ZLL)(Pos, 0)=Config.Info_Get(KindOfStream)[Pos][0];
    ZLL->Write(Config.Language_Get(Config.Info_Get(KindOfStream, 2, Info_Text)), 2, Info_Text);

    //Special cases
    if (KindOfStream==Stream_General)
        General_Fill();
    else
        ZLL->Write(Ztring::ToZtring(Stream[KindOfStream]->size()), 3, Info_Text);

    KindOfStream_Last=KindOfStream;
    StreamPos_Last=Stream[KindOfStream]->size()-1;

    return Stream[KindOfStream]->size()-1; //The position in the stream count
}

//***************************************************************************
// Filling
//***************************************************************************

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const Ztring &Value)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter))=Value;
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const char* Value, intu Value_Size, bool Utf8)
{
    if (Utf8) (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Local(Value, Value_Size);
    else      (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_UTF8 (Value, Value_Size);
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, const wchar_t* Value, intu Value_Size)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Unicode(Value, Value_Size);
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int16u Value, intu Radix)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Number(Value, Radix);
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int32u Value, intu Radix)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Number(Value, Radix);
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int32s Value, intu Radix)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Number(Value, Radix);
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, int64u Value, intu Radix)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Number(Value, Radix);
}

//---------------------------------------------------------------------------
void File__Base::Fill (stream_t StreamKind, size_t StreamPos, const char* Parameter, float32 Value, intu AfterComma, ztring_t Options)
{
    (*Stream[StreamKind])[StreamPos](Ztring().From_UTF8(Parameter)).From_Number(Value, AfterComma, Options);
}

//---------------------------------------------------------------------------
void File__Base::Fill_HowTo (stream_t StreamKind, size_t StreamPos, const char* Parameter, const char* Value)
{
    //TODO
}

//***************************************************************************
// Divers
//***************************************************************************

void File__Base::Clear()
{
    for (size_t StreamKind=0; StreamKind<Stream_Max; StreamKind++)
    {
        (*Stream[StreamKind]).clear();
        Optimized[StreamKind].clear();
    }
}

//---------------------------------------------------------------------------
int File__Base::Read()
{
    return -1;
}

//---------------------------------------------------------------------------
int File__Base::Read(const int8u* Begin, size_t Begin_Size, const int8u* End, size_t End_Size, int64u FileSize)
{
    return -1;
}

//---------------------------------------------------------------------------
int File__Base::Write(stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &ToSet, const Ztring &OldValue)
{
    return -1;
}

int File__Base::WriteToDisk()
{
    return -1;
}

//---------------------------------------------------------------------------
void File__Base::HowTo(stream_t StreamKind)
{
    return;
}

//---------------------------------------------------------------------------
void File__Base::Merge(const File__Base &ToAdd)
{
    for (size_t StreamKind=0; StreamKind<(size_t)Stream_Max; StreamKind++)
        for (size_t StreamPos=0; StreamPos<ToAdd.Stream[StreamKind]->size(); StreamPos++)
        {
            if (Stream[StreamKind]->size()<=StreamPos)
                Stream_Prepare((stream_t)StreamKind);
            for (size_t ToAdd_Pos=0; ToAdd_Pos<ToAdd.Stream[StreamKind]->operator[](StreamPos).size(); ToAdd_Pos++)
            {
                if (ToAdd.Stream[StreamKind]->operator[](StreamPos)[ToAdd_Pos].size()>Info_Text)
                {
                    Ztring Z1=ToAdd.Stream[StreamKind]->operator[](StreamPos)[ToAdd_Pos][Info_Text];
                    if (!Z1.empty())
                    {
                        size_t Pos=(*Stream[StreamKind])[StreamPos].Find(ToAdd.Stream[StreamKind]->operator[](StreamPos)[ToAdd_Pos][Info_Name]);
                        if (Pos!=Error && (*Stream[StreamKind])[StreamPos](Pos, Info_Text).empty())
                            (*Stream[StreamKind])[StreamPos].Write(Z1, Pos, Info_Text);
                    }
                }
            }
        }
}

} //NameSpace


