// File_Mpega - Info for MPEG Audio files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_MPEGA_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_MPEGA_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
    #pragma hdrstop
#endif
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#if defined(MEDIAINFO_ID3_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_ID3_NO))
    #include "MediaInfo/Audio/File_Id3.h"
#endif
#include "MediaInfo/Audio/File_Mpega.h"
#include "ZenLib/BitStream.h"
#include <ZenLib/Utils.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
const Char* Mpega_Version[4]=
{
    _T("MPEG-2.5"),
    _T(""),
    _T("MPEG-2"),
    _T("MPEG-1")
};
const Char* Mpega_Version_String[4]=
{
    _T("MPEG-2.5 Audio"),
    _T(""),
    _T("MPEG-2 Audio"),
    _T("MPEG-1 Audio")
};
const Char* Mpega_Layer[4]=
{
    _T(""),
    _T(" L3"),
    _T(" L2"),
    _T(" L1")
};
const Char* Mpega_Layer_String[4]=
{
    _T(""),
    _T(" layer 3"),
    _T(" layer 2"),
    _T(" layer 1")
};
const size_t Mpega_BitRate[4][4][16]=
{
    {{0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},  //MPEG Audio 2.5 layer X
     {0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160,   0},  //MPEG Audio 2.5 layer 3
     {0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160,   0},  //MPEG Audio 2.5 layer 2
     {0,  32,  48,  56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256,   0}}, //MPEG Audio 2.5 layer 1
    {{0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},  //MPEG Audio X layer X
     {0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},  //MPEG Audio X layer 3
     {0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},  //MPEG Audio X layer 2
     {0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0}}, //MPEG Audio X layer 1
    {{0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},  //MPEG Audio 2 layer X
     {0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160,   0},  //MPEG Audio 2 layer 3
     {0,   8,  16,  24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160,   0},  //MPEG Audio 2 layer 2
     {0,  32,  48,  56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256,   0}}, //MPEG Audio 2 layer 1
    {{0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0},  //MPEG Audio 1 layer X
     {0,  32,  40,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320,   0},  //MPEG Audio 1 layer 3
     {0,  32,  48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384,   0},  //MPEG Audio 1 layer 2
     {0,  32,  64,  96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448,   0}}  //MPEG Audio 1 layer 1
};
const size_t Mpega_SamplingRate[4][4]=
{
    {11025, 12000,  8000, 0}, //MPEG Audio 2.5
    {    0,     0,     0, 0}, //MPEG Audio X
    {22050, 24000, 16000, 0}, //MPEG Audio 2
    {44100, 48000, 32000, 0}  //MPEG Audio 1
};
const size_t Mpega_Channels[4]=
{
    2,
    2,
    2,
    1
};
const Char* Mpega_Codec_Profile[4]=
{
    _T(""),
    _T("Joint stereo"),
    _T("Dual mono"),
    _T("")
};
//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

size_t Mpega_FrameSize(int32u Header);

//---------------------------------------------------------------------------
int File_Mpega::Read(const int8u* Begin_, size_t Begin_Size_, const int8u* End_, size_t End_Size_, int64u FileSize)
{
    Begin=Begin_;
    Begin_Size=Begin_Size_;
    End=End_;
    End_Size=End_Size_;
    Offset=0;

    //Integrity
    if (Begin_Size<128)
        return -1;

    //Init
    FrameSize=0;
    VBR_Frames=0;
    VBR_FileSize=(int32u)FileSize;

    //ID3v1 and ID3 v2 present?
    size_t Id3v1_Size=0;
    size_t Id3v2_Size=0;
    if (CC3(Begin)==CC3("ID3"))
    {
        //ID3v2, must skip it here
        Id3v2_Size=(Begin[6]<<21) | (Begin[7]<<14) | (Begin[8]<<7) | (Begin[9]<<0)+10;
        Offset+=Id3v2_Size;
        if (Offset+4>=Begin_Size)
            return -1;
    }
    if (End_Size>=128 && CC3(End+End_Size-128)==CC3("TAG"))
        //ID3v1
        Id3v1_Size=128;

    //Search for header
    if (NextFrame()==Error)
        return -1;

    // AAAAAAAA AAABBCCD EEEEFFGH IIJJKLMM {ZZZZ}
    //
    // Label (Position) Description
    // A (31-21) Frame sync
    //           All bits set (1)
    // B (20,19) MPEG Audio version ID
    //           00 - MPEG Ver 2.5, 01 - reserved, 10 - Ver 2, 11 - Ver 1
    //           Note: MPEG Ver 2.5 is not official; bit # 20 indicates 2.5
    // C (18,17) Layer description
    //           00 - reserved, 01 - Layer III, 10 - Layer II, 11 - Layer I
    // D    (16) Protection bit
    //           0 - None, 1 - Protected by CRC (16bit crc follows header)
    // E (15,12) Bitrate index
    // F (11,10) Sampling rate
    // G     (9) Padding bit
    //           0 - frame not padded, 1 - frame padded with one extra slot
    // H     (8) Private bit
    //           0 - not private, 1 - private
    //           Note: May be freely used for other needs of an application.
    // I   (7,6) Channel Mode
    //           00 - Stereo, 01 - Joint stereo, 10 - Dual (Stereo), 11 - Mono
    // J   (5,4) Mode extension (Only if Joint stereo)
    //           Used to join data; bits dynamically generated by an encoder.
    // K     (3) Copyright
    //           0 - Audio is not copyrighted, 1 - Audio is marked copyrighted
    // L     (2) Original
    //           0 - Copy of original media, 1 - Original media 
    // M   (1,0) Emphasis
    //           00 - none, 01 - 50/15 ms, 10 - reserved, 11 - CCIT J.17
    // Z (32-35) CRC  !!OPTIONAL!!
    //           Note: NOT part of header, just appended on end when needed
    //
    BitStream BS(Begin+Offset, Begin_Size-Offset);
    BS.Skip(11); //Frame Sync, All bits set (1)
    Mpeg=BS.Get(2); //Version ID
    Layer=BS.Get(2); //Layer ID
    BS.Skip(1); //Protection
    size_t BitRate=BS.Get(4); //BitRate
    size_t SamplingRate=BS.Get(2); //SamplingRate
    BS.Skip(1); //Padding
    BS.Skip(1); //Private
    Channels=BS.Get(2); //Channel Mode

	//VBR headers
    while (Header_Xing()!=Error || Header_VBRI()!=Error)
        if (NextFrame()==Error && !(Offset==Begin_Size || Offset+1==Begin_Size)) //Test if this is not the end of the file, only one frame (and this frame can be malformed)
            return -1;
    Ztring BitRate_Mode;
    if (VBR_Frames==0)
    {
        BitRate=Mpega_BitRate[Mpeg][Layer][BitRate]*1000;
        BitRate_Mode=_T("CBR");
    }
    else
    {
        float32 FrameLength=(VBR_FileSize?VBR_FileSize:FileSize)/VBR_Frames;
        size_t Divider;
        if (Mpeg==3 && Layer==3) //MPEG 1 Layer 1
             Divider=384/8;
        else if ((Mpeg==2 || Mpeg==0) && Layer==1) //MPEG 2 or 2.5 Layer 3
            Divider=576/8;
        else
            Divider=1152/8;
        BitRate=FrameLength*Mpega_SamplingRate[Mpeg][SamplingRate]/Divider;
        BitRate_Mode=_T("VBR");
    }

	//Coherancy test
    size_t Loop_Count=32;
    size_t Mp3Header_NOK=0;
    while (Mp3Header_NOK==0 && Loop_Count && Offset+FrameSize+4<Begin_Size)
    {
        Offset+=FrameSize;
        Mp3Header_NOK=NextFrame();
        if (Mp3Header_NOK==1) //Some malformed MPEG audio
            Mp3Header_NOK=0;
        if (Mp3Header_NOK==0)
            Loop_Count--;
        if (Offset+FrameSize+128==Begin_Size && CC3(Begin+Offset+FrameSize)==CC3("TAG"))
            Offset+=128; //Special case : Id3v1 is at the end of Begin (small file)
    }
    if (Mp3Header_NOK)
        return -1;

    //Filling
    Stream_Prepare(Stream_General);
    Fill("Format", "MPEG Audio");
    Stream_Prepare(Stream_Audio);
    Fill("Codec", Ztring(Mpega_Version[Mpeg])+Mpega_Layer[Layer]);
    Fill("Codec/String", Ztring(Mpega_Version_String[Mpeg])+Mpega_Layer_String[Layer]);
    Fill("SamplingRate", Mpega_SamplingRate[Mpeg][SamplingRate]);
    Fill("Channel(s)", Mpega_Channels[Channels]);
    Fill("Codec_Profile", Mpega_Codec_Profile[Channels]);
    Fill("BitRate", BitRate);
    Fill("Resolution", 16);
    Fill("BitRate_Mode", BitRate_Mode);
    Fill("Encoded_Library", Encoded_Library);

    //if (BitRate>0)
    //    General[0](_T("PlayTime")).From_Number(float32_int32s(int64u_float32((VBR_FileSize? VBR_FileSize : General[0](_T("FileSize")).To_int64u())) *1000.0f/ (int64u_float32(Audio[0](_T("BitRate")).To_int64u()) / 8.0f) ));

    //Tags
    #if defined(MEDIAINFO_ID3_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_ID3_NO))
        if (Id3v1_Size || Id3v2_Size)
        {
            File_Id3 Id3;
            if (Id3.Open(CompleteFileName)<0)
                Id3.Open(Begin, Id3v2_Size, End+End_Size-Id3v1_Size, Id3v1_Size);
            Merge(Id3);
        }
    #endif //ID3

	return 1;
}

//---------------------------------------------------------------------------
int File_Mpega::Write (const Ztring &ToSet, stream_t StreamKind, size_t StreamNumber, const Ztring &Parameter, const Ztring &OldValue)
{
    return -1;
}

//---------------------------------------------------------------------------
void File_Mpega::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("ChannelPositions"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
    }
}

//---------------------------------------------------------------------------
size_t File_Mpega::NextFrame()
{
	size_t Level=0;
	size_t Offset_Header=Offset;
    size_t Malformed=0;
	while (Offset_Header+4<Begin_Size)
    {
		if (Level==3)
        {
		}
		     if (Level==2)
        {
			//Reset the offset
            Offset_Header-=2;
            Offset_Header-=Malformed;

            //FrameSize
          	int32u Header=BigEndian2int32u(Begin+Offset_Header);
            FrameSize=Mpega_FrameSize(Header);
            if (FrameSize==Error)
                return Error;
                
            //Junk
            size_t Junk=Offset_Header-Offset;
            if (Junk==Error) //Malformed
                Junk=0;

            //Return value
            Offset=Offset_Header;
            return Junk;
		}
		else if (Level==1)
        {
			if ((Begin[Offset_Header]&0xE0)==0xE0
             && (Begin[Offset_Header]&0x18)!=0x08
             && (Begin[Offset_Header]&0x06)!=0x00)
				Level=2;
            else
				Level=0;
		}
		else if (Level==0)
        {
			if (Begin[Offset_Header]==0xFF)
				Level=1;
            //MPEG Audio malformed! Sometimes the padding bit is set but the padding is not here.
            else if (Offset_Header>0
                  && Begin[Offset_Header-1]==0xFF //Level 0
                  && (Begin[Offset_Header]&0xE0)==0xE0 //Level 1
                  && (Begin[Offset_Header]&0x18)!=0x08
                  && (Begin[Offset_Header]&0x06)!=0x00)
            {
                Malformed=1;
				Level=2;
            }
		}
        Offset_Header++;
	}
    return Error;
}

//---------------------------------------------------------------------------
size_t File_Mpega::Header_Xing()
{
    int32u Xing_Header_Offset;
    if (Mpeg==3) //MPEG-1
        if (Channels==3) //Mono
            Xing_Header_Offset=21;
        else
            Xing_Header_Offset=36;
    else //MPEG-2 or 2.5
        if (Channels==3) //Mono
            Xing_Header_Offset=13;
        else
            Xing_Header_Offset=21;
    if (Offset+Xing_Header_Offset+128<Begin_Size)
    {
        const int8u* Xing_Header=Begin+Offset+Xing_Header_Offset;
        if (CC4(Xing_Header)==CC4("Xing") || CC4(Xing_Header)==CC4("Info"))
        {
            size_t Xing_Flags=0;
            Xing_Header+=4; // "Xing"
            Xing_Flags=BigEndian2int32s((char*)Xing_Header);
            Xing_Header+=4; //Flags
            int32u Xing_Header_Size=8
                                   +((Xing_Flags&0x1) ? 4:0)    //FrameCount
                                   +((Xing_Flags&0x2) ? 4:0)    //FileSize
                                   +((Xing_Flags&0x4) ? 100:0)  //TOC
                                   +((Xing_Flags&0x8) ? 4:0)    //Scale
                                   +((Xing_Flags&0x10)? 348:0); //Lame

            if (Xing_Header_Offset+Xing_Header_Size<Begin_Size)
            {
                if (Xing_Flags&0x1) //FrameCount
                {
                    VBR_Frames=BigEndian2int32s((char*)Xing_Header);
                    Xing_Header+=4;
                }

                if (Xing_Flags&0x2) //FileSize
                {
                    VBR_FileSize=BigEndian2int32s((char*)Xing_Header);
                    Xing_Header+=4;
                }

                if (Xing_Flags&0x4) //TOC
                    Xing_Header+=100;

                if (Xing_Flags&0x8) //Scale
                    Xing_Header+=4;

                if (Xing_Flags&0xF) //LAME
                {
                    Encoded_Library.From_Local((char*)Xing_Header, 9);
                    if (Encoded_Library>=_T("LAME3.90") || Encoded_Library<_T("LAME9.99"))
                    {
                        if ((CC1(Xing_Header+9)&0x0F)==1 || (CC1(Xing_Header+9)&0x0F)==8) //2 possible values for CBR
                            VBR_Frames=0;
                        //TODO : Lame tag, http://gabriel.mp3-tech.org/mp3infotag.html
                    }
                    else
                        Encoded_Library.From_Local((char*)Xing_Header, 20); //Long tag version, if version<3.90
                }
            }

            if (CC4(Xing_Header)==CC4("Info"))
                VBR_Frames=0; //This is not a VBR file

            Offset+=FrameSize;
            return 1;
        }
    }
    return Error;
}

//---------------------------------------------------------------------------
size_t File_Mpega::Header_VBRI()
{
    const size_t Fraunhofer_Header_Offset=36;
    if (Offset+Fraunhofer_Header_Offset+32<Begin_Size)
    {
        const int8u* Fraunhofer_Header=Begin+Offset+Fraunhofer_Header_Offset;
        if (CC4(Fraunhofer_Header)==CC4("VBRI") && CC2(Fraunhofer_Header+4)==0x0001) //VBRI v1 only
        {
            VBR_FileSize=BigEndian2int32s(Fraunhofer_Header+10);
            VBR_Frames=BigEndian2int32s(Fraunhofer_Header+14);
            Offset+=FrameSize;
            return 1;
        }
    }
    return Error;
}

//---------------------------------------------------------------------------
size_t Mpega_FrameSize(int32u I0)
{
    size_t Mpeg=(I0>>19)&0x3;
    size_t Layer=(I0>>17)&0x3;
    size_t BitRate=(I0>>12)&0xF;
    size_t SamplingRate=(I0>>10)&0x3;
    size_t Padded=(I0>>9)&0x1;
	size_t FrameSize=0;

	if (Mpega_SamplingRate[Mpeg][SamplingRate]==0)
        return Error;

    if (Layer==3) //Layer 1
    {
        if (Mpeg==3) //MPEG-1
            FrameSize=(12*Mpega_BitRate[Mpeg][Layer][BitRate]*1000/Mpega_SamplingRate[Mpeg][SamplingRate]+Padded)*4;
        else //MPEG-2 and 2.5
            FrameSize=(6*Mpega_BitRate[Mpeg][Layer][BitRate]*1000/Mpega_SamplingRate[Mpeg][SamplingRate]+Padded)*4;
    }
    else //Layer 2 and 3
    {
        if (Mpeg==3) //MPEG-1
            FrameSize=144*Mpega_BitRate[Mpeg][Layer][BitRate]*1000/Mpega_SamplingRate[Mpeg][SamplingRate]+Padded;
        else //MPEG-2 and 2.5
            FrameSize=72*Mpega_BitRate[Mpeg][Layer][BitRate]*1000/Mpega_SamplingRate[Mpeg][SamplingRate]+Padded;
    }

	return FrameSize;
}

} //NameSpace

#endif //MEDIAINFO_MPEG_*

