// File_Ac3 - Info for AC3 files
// Copyright (C) 2004-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// AC3 - Format
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//  StartCode                        2 bytes - 0x0B77
//  CRC                              2 bytes
//  Sampling rate                    2 bits
//  Bit rate / word per syncframe    6 bits
//  BitStream ID                     5 bits - 01000
//  BitStream Mode (FX, voice...)    3 bits
//  Channels                         3 bits
//  (option) Central amplification   2 bits - if Channels & 001 && Channels!=1
//  (option) Back amplification      2 bits - if Channels & 100
//  (option) Dolby                   2 bits - if Channels == 2
//  Sub woofer present               1 bit
//  ...
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_AC3_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_AC3_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Audio/File_Ac3.h"
#include <ZenLib/Utils.h>
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
const size_t AC3_SamplingRate[]=
{ 48000,  44100,  32000,      0,};
const size_t AC3_BitRate_Size=19;
const size_t AC3_BitRate[]=
{  32000,   40000,   48000,   56000,   64000,   80000,   96000,  112000,
  128000,  160000,  192000,  224000,  256000,  320000,  384000,  448000,
  512000,  576000,  640000};
const size_t AC3_Channels[]=
{2, 1, 2, 3, 3, 4, 4, 5};
const Char*  AC3_ChannelPositions[]=
{
    _T("L R"),
    _T("C"),
    _T("L R"),
    _T("L C R"),
    _T("Front: L R,   Rear: C"),
    _T("Front: L C R, Rear: C"),
    _T("Front: L R,   Rear: L R"),
    _T("Front: L C R, Rear: L R")
};

//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

//---------------------------------------------------------------------------
int File_Ac3::Read(const int8u* Begin, size_t Begin_Size, const int8u* End, size_t End_Size, int64u FileSize)
{
    if (Begin_Size<=4)
        return -2;

    size_t Offset=0;
    while (Offset<=Begin_Size-4 && !(CC2(Begin+Offset)==0x0B77))
        Offset++;
    if (Offset>Begin_Size-4)
        return -1;

    //Coherancy test : for example, to not find AC3 in a file with some "0B77" block
    size_t Offset_Save=Offset;
    size_t Loop_Count=Begin_Size/10000;
    if (Loop_Count>4)
        Loop_Count=4;
    for (size_t Loop=0; Loop<=Loop_Count; Loop++)
    {
        while (Offset<=Begin_Size-4 && !(CC2(Begin+Offset)==0x0B77))
            Offset++;
        if (Offset>Begin_Size-4)
            return -1;
        Offset++;
    }
    if (Offset-Offset_Save>0x4000)
        return -1;//Suspect there is not enougth "0B77" in the file to be AC3
    Offset=Offset_Save;

    //Processing
    int32u I0=BigEndian2int32u((char*)Begin+Offset+4);
    size_t SamplingRate=I0>>30;
    size_t BitRate=(I0&0x3E000000)>>25;
    if (BitRate>=AC3_BitRate_Size)
        return -1;
    size_t Channels=(I0&0x0000E000)>>13;

    size_t Decalage=0; bool Dolby=false;
    if (Channels&0x01 && Channels!=0x01) Decalage+=2;
    if (Channels&0x04) Decalage+=2;
    if (Channels==0x02) // Dolby Digital?
    {
        size_t DD=(I0>>(12-Decalage))&0x03;
        if (DD==0x02)
            Dolby=true;
        Decalage+=2;
    }
    int SubWoofer=(I0>>(12-Decalage))&0x01;

    //Filling
    Stream_Prepare(Stream_General);
    General[0](_T("Format"))=_T("AC3");
    General[0](_T("Format/String"))=_T("AC3 (Audio Coding 3)");
    General[0](_T("Format/Extensions"))=_T("AC3");

    Stream_Prepare(Stream_Audio);
    Audio[0](_T("Codec"))=_T("AC3");
    Audio[0](_T("SamplingRate")).From_Number(AC3_SamplingRate[SamplingRate]);
    Audio[0](_T("BitRate")).From_Number(AC3_BitRate[BitRate]);
    Audio[0](_T("Channel(s)")).From_Number(AC3_Channels[Channels]);
    if (Channels==0)
        Audio[0](_T("Codec_Profile"))=_T("Dual Mono");
    Audio[0](_T("ChannelPositions"))=AC3_ChannelPositions[Channels];
    if (SubWoofer)
    {
        Audio[0](_T("Channel(s)")).From_Number(AC3_Channels[Channels]+1);
        Audio[0](_T("ChannelPositions"))+=_T(", Subwoofer");
    }
    if (Dolby)
        Audio[0](_T("ChannelPositions"))+=_T(" (Dolby Digital)");
    Audio[0](_T("BitRate_Mode"))=_T("CBR");

    //AC3 is CBR --> We can calculate PlayTime
    General_Fill(); //To feed PlayTime
    if (!General[0](_T("FileSize")).empty())
        General[0](_T("PlayTime")).From_Number(General[0](_T("FileSize")).To_int64u()*8/AC3_BitRate[BitRate]);
    return 1;
}

//---------------------------------------------------------------------------
void File_Ac3::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("BitRate_Mode"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("ChannelPositions"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
        Audio[0](_T("Resolution"), Info_HowTo)=_T("N");
    }
}

} //NameSpace

#endif //MEDIAINFO_AC3_*

