// File_Aac - Info for AAC files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_AAC_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_AAC_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include "MediaInfo/Audio/File_Aac.h"
#if defined(MEDIAINFO_ID3_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_ID3_NO))
    #include "MediaInfo/Audio/File_Id3.h"
#endif
#include "ZenLib/Utils.h"
#include "ZenLib/BitStream.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//---------------------------------------------------------------------------
const size_t AAC_SamplingRate[]=
{96000, 88200, 64000, 48000, 44100, 32000, 24000, 22050,
 16000, 12000, 11025,  8000, Error, Error, Error, Error,};
//---------------------------------------------------------------------------

//***************************************************************************
// Functions
//***************************************************************************

//---------------------------------------------------------------------------
int File_Aac::Read (const int8u* Begin, size_t Begin_Size, const int8u* End, size_t End_Size, int64u FileSize)
{
    if (Begin_Size<16)
        return -2;

    size_t Offset=0;
    size_t BitRate=Error;
    size_t Mpeg=Error;
    size_t Profile=Error;
    size_t SamplingRate=Error;
    size_t Channels=Error;
    bool VBR=false;
    bool CBR=false;

    //ID3v1 and ID3 v2 present?
    size_t Id3v1_Size=0;
    size_t Id3v2_Size=0;
    if (CC3(Begin)==CC3("ID3"))
    {
        //ID3v2, must skip it here
        Id3v2_Size=(Begin[6]<<21) | (Begin[7]<<14) | (Begin[8]<<7) | (Begin[9]<<0)+10;
        Offset+=Id3v2_Size;
        if (Offset+4>=Begin_Size)
            return -1;
    }
    if (End_Size>=128 && CC3(End+End_Size-128)==CC3("TAG"))
        //ID3v1
        Id3v1_Size=128;

    if (Offset>Begin_Size-16)
        return -1;

    //Check if ADIF header
    if (CC4(Begin)==CC4("ADIF"))
    {
        BitStream BS(Begin+Offset, Begin_Size-Offset);
        BS.Skip(32); //ID: "ADIF"
        bool Copyright_Present=(bool)BS.Get(1); //Copyright_Present
        if (Copyright_Present)
            {BS.Skip(32); BS.Skip(32); BS.Skip(8);} //Copyright_ID
        BS.Skip(1); //Original
        BS.Skip(1); //Home
        bool BitStream_Type=BS.Get(1); //BitStream_Type, 0=CBR, 1=VBR
        BitRate=BS.Get(23); //For CBR: bitrate, for VBR: peak bitrate, 0 means unknown
        if (BitStream_Type==0)
            CBR=true;
        else
            VBR=true;
        //BS.Skip(4); //num_program_config_elements
		//the next 2 fields come (num_program_config_elements+1) times
        //buffer_fullness 	20 	only if bitstream_type == 0
        //program_config_element 	VAR
    }

    //Check if ADTS header
    else if ((CC2(Begin+Offset)&0xFFF6)==0xFFF0) //12 bits + ID + 2 bits
    {
        BitStream BS(Begin+Offset, Begin_Size-Offset);
        BS.Skip(12); //syncword, 0xFFF
        Mpeg=BS.Get(1); //id: 0=MPEG-4, 1=MPEG-2
        BS.Skip(2); //layer, should be 0
        bool Protection_Absent=(bool)BS.Get(1); //protection_absent
        Profile=BS.Get(2); //Profile
        size_t SamplingRate_Index=BS.Get(4); //SamplinRate
        BS.Skip(1); //Private
        Channels=BS.Get(3); //Channels
        BS.Skip(1); //Original
        BS.Skip(1); //Home
        //if ID=0 and "emphasis", here should be 2 bits more. To be confirmed, and how Can I get "emphasis"?
        BS.Skip(1); //Copyright_id_bit
        BS.Skip(1); //Copyright_id_start
        size_t Frame_Length=BS.Get(13); //Frame length
        size_t Fullness=BS.Get(11); //Buffer_Fullness
        if (Fullness==0x7FF)
            VBR=true;
        BS.Skip(2); //No_raw_blocks_in_frame
        if (Protection_Absent)
            BS.Skip(16); //CRC

        //Filling
        SamplingRate=AAC_SamplingRate[SamplingRate_Index];
        Offset+=Frame_Length;
        //size_t FramePerSecond=(float)SamplingRate/1024; //For raw AAC, there are always 1024 samples per frame (but 960 in MP4?)
        //size_t BytesPerFrame=FileSize/(float)(FrameCount*1000);
        //BitRate=8*BytesPerFrame*FramePerSecond+0.5);
    }

    //AAC header
    else if (CC3(Begin)==CC3("AAC"))
    {
        //What is it???
    }
    else
        return -1;

    //Filling
    Stream_Prepare(Stream_General);
    Fill ("Format", "AAC");

    Stream_Prepare(Stream_Audio);
    Fill ("Codec", "AAC");
    if (Channels!=Error) Fill("Channel(s)", Channels);
    if (BitRate!=Error)  Fill("BitRate", BitRate);
    if (CBR)             Fill("BitRate_Mode", "CBR");
    if (VBR)             Fill("BitRate_Mode", "VBR");

    Ztring Family, ProfileS;
    if (Mpeg==0)    Family  +=_T("MPEG-4");
    else            Family  +=_T("MPEG-2");
                    Family  +=_T(" AAC ");
    if (Profile==0) ProfileS+=_T("Main");
    if (Profile==1) ProfileS+=_T("LC");
    if (Profile==2) ProfileS+=_T("SSR");
    if (Profile==3) ProfileS+=_T("LTP");
    if (SamplingRate<=24000)
    {
        SamplingRate*=2;
        ProfileS+=_T("-SBR");
    }
    Fill("Codec/Family",  Family);
    Fill("Codec_Profile", ProfileS);
    if (SamplingRate!=Error) Fill("SamplingRate", SamplingRate);

    //Tags
    #if defined(MEDIAINFO_ID3_YES) || (!defined(MEDIAINFO_AUDIO_NO) && !defined(MEDIAINFO_ID3_NO))
        if (Id3v1_Size || Id3v2_Size)
        {
            File_Id3 Id3;
            if (Id3.Open(CompleteFileName)<0)
                Id3.Open(Begin, Id3v2_Size, End+End_Size-Id3v1_Size, Id3v1_Size);
            Merge(Id3);
        }
    #endif //ID3

    return 1;
}

//---------------------------------------------------------------------------
void File_Aac::HowTo(stream_t StreamKind)
{
        if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Resolution"), Info_HowTo)=_T("N");
    }
}

} //NameSpace

#endif //MEDIAINFO_AAC_*

