/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Log: TimeStamp.java,v $
 * Revision 1.5  2011/02/14 22:32:49  vizigoth
 * First commit after major sourceforge outage.
 *
 * Revision 1.4  2011/01/05 13:09:06  vizigoth
 * Created new forge for making record and union type values.
 *
 * Revision 1.3  2009/05/14 16:15:24  vizigoth
 * Major refactor to remove dependency on JPA and introduce better interface and implementation separation. Removed all setPropertiesFromInterface and castFromInterface methods.
 *
 * Revision 1.2  2007/12/14 15:01:49  vizigoth
 * Added to and edited document comments to a release level.
 *
 * Revision 1.1  2007/11/13 22:12:59  vizigoth
 * Public release of MAJ API.
 */

package tv.amwa.maj.record;

import java.util.Calendar;


/** <p>Specifies a date and time in UTC (Coordinated Universal Time). The value is made
 * up of a {@linkplain DateStruct date} and {@linkplain TimeStruct time} structures.</p>
 * 
 * <p>For more information on UTC, see the <a href="http://en.wikipedia.org/wiki/UTC">entry on
 * Coordinated Universal Time on Wikipedia.</a></p>
 * 
 * <p>To make values of this type, use the following methods from the 
 * {@linkplain tv.amwa.maj.industry.Forge MAJ forge}:</p>
 * 
 * <ul>
 *  <li>From its component parts: 
 *  {@link tv.amwa.maj.industry.Forge#makeTimeStamp(DateStruct, TimeStruct)} and
 *  {@link tv.amwa.maj.industry.Forge#makeTimeStamp(short, byte, byte, byte, byte, byte, byte)};</li>
 *  <li>From a {@link java.util.Calendar} value:
 *  {@link tv.amwa.maj.industry.Forge#makeTimeStamp(java.util.Calendar)};</li>
 *  <li>The date and time now: {@link tv.amwa.maj.industry.Forge#now()};</li>
 *  <li>From a string representation as generated by {@link #toString()}: 
 *  {@link tv.amwa.maj.industry.Forge#parseTimeStamp(String)}.</li>
 * </ul>  
 * 
 * @see TimeStruct
 * @see DateStruct
 * @see tv.amwa.maj.industry.TypeDefinitions#TimeStamp
 * 
 *
*/

public interface TimeStamp { 

	/**
	 * <p>Simultaneously sets the date and time components of the time stamp.</p>
	 *
	 * @param date Date component of the time stamp.
	 * @param time Time component of the time stamp.
	 * 
	 * @throws NullPointerException One or both of the given data or time structures are <code>null</code>.
	 * @throws IllegalArgumentException One or both of the given date or time values are not valid. 
	 */
	public void setStamp(
			DateStruct date, 
			TimeStruct time)
		throws NullPointerException,
			IllegalArgumentException;
	
	/**
	 * <p>Returns the date component of the time stamp.</p>
	 *
	 * @return Date component of the time stamp.
	 */
	public DateStruct getDate();

	/**
	 * <p>Sets the date component of the time stamp.</p>
	 *
	 * @param date Date component of the time stamp.
	 * 
	 * @throws NullPointerException The given data structure is <code>null</code>.
	 * @throws IllegalArgumentException The given date is not valid.
	 */
	public void setDate(
			DateStruct date)
		throws NullPointerException, 
			IllegalArgumentException;

	/**
	 * <p>Returns the time component of the time stamp.</p>
	 *
	 * @return Time component of the time stamp.
	 */
	public TimeStruct getTime();

	/**
	 * <p>Sets the time component of the time stamp.</p>
	 *
	 * @param time Time component of the time stamp.
	 * 
	 * @throws NullPointerException The given time structure is <code>null</code>.
	 * @throws IllegalArgumentException The given time is not valid.
	 */
	public void setTime(
			TimeStruct time)
		throws NullPointerException, 
			IllegalArgumentException;

	/**
	 * <p>Create a cloned copy of this time stamp.</p>
	 * 
	 * @return Cloned copy of this time stamp.
	 */
	public TimeStamp clone();
	
	/** 
	 * <p>Formats the time stamp as a string according ISO 8601 and as a UTC value.</p>
	 * 
	 * <p>The format of the string is "<code>yyyy-MM-dd'T'HH:mm:ss.SSZ</code>", where:</p>
	 * 
	 * <ul>
	 *  <li>yyyy - Year part of the time stamp value.</li>
	 *  <li>MM - Month part of the time stamp value, in the range 1 to 12.</li>
	 *  <li>dd - Day of the month part of the time stamp value, in the range 1 to 31.</li>
	 *  <li>HH - Hour of the day of the time stamp value, specified in 24-hour clock in 
	 *  the range 0 to 23.</li>
	 *  <li>mm - Minute part of the time stamp value, in the range 0 to 59.</li>
	 *  <li>ss - Second part of the time stamp value, normally in the range 0 to 59.</li>
	 *  <li>SS - hundreths of a second part of the time stamp value, in the range .00 to .99.</li>
	 *  <li>Z - The letter "Z" to indicate that this value is shown in UTC.</li>
	 * </ul>
	 * 
	 * <p>The representation of a time stamp generated by this method can be converted back to
	 * a time stamp value using {@link tv.amwa.maj.industry.Forge#parseTimeStamp(String)}.</p>
	 * 
	 * @see tv.amwa.maj.industry.Forge#parseTimeStamp(String)
	 * @see java.text.SimpleDateFormat
	 */
	public String toString();
	
	/**
	 * <p>Returns a calendar value representing this time stamp. The method returns a copy
	 * of internal calendar value to preserve the internal state of this value.</p>
	 *
	 * @return Calendar value representing this time stamp.
	 * 
	 * @see tv.amwa.maj.industry.Forge#makeTimeStamp(Calendar)
	 */
	public Calendar getTimeStamp();

}
