/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package tv.amwa.maj.industry;

/**
 * <p>Specifies a media class with metadata that can be persisted using the
 * <a href="http://en.wikipedia.org/wiki/Java_Persistence_API">Java Persistence API</a>.
 * All media entities must provide a Java long persistent identifier and an index that is used
 * to order values if the entity is persisted as part of a {@linkplain StrongReferenceVector strong
 * reference vector}.</p>
 *
 * <p>The implementing class or one of its superclasses must provide fields "<code>persistentID</code>"
 * and "<code>persistentIndex</code>" so that the JPA implementation can create and manipulate
 * objects as appropriate. Here is an example of the required code:</p>
 *
 * <pre>
 *     private Long persistentID = null;
 *     private int persistentIndex = 0;
 *
 *     public Long getPersistentID() {
 *
 *         return persistentID;
 *     }
 *
 *     public void setPersistentIndex(
 *         int index) {
 *
 *         this.persistentIndex = index;
 *     }
 * </pre>
 *
 * <p>Once these fields are provided and the methods are implemented, entities can be managed
 * by an appropriately configured entity manager. The configuration is generated by the
 * {@link JPAGenerator#generateORM(java.util.Collection, String)} method.</p>
 *
 *
 *
 * @see tv.amwa.maj.meta.TypeDefinitionStrongObjectReference
 * @see tv.amwa.maj.meta.TypeDefinitionVariableArray
 * @see JPAGenerator#generateORM(java.util.Collection, String)
 */
public interface MediaEntity
	extends MetadataObject {

	/**
	 * <p>Returns the persistent identifier for this media entity. This will be automatically
	 * generated by the persistence provider when the object is persisted for the first time.
	 * Before this, the identifier will be null.</p>
	 *
	 * @return Persistent identifier for this media entity.
	 */
	public Long getPersistentID();

	/**
	 * <p>If the media entity is an element of a strong reference vector, set the index of
	 * that element in the vector just before the value is persisted. When the value is
	 * retrieved, the index will be used to restore the item to the correct position in
	 * its list. This is required to overcome a shortcoming of the Java Persistence API
	 * when persisting ordered collections.</p>
	 *
	 * @param index Index of the entity within the list that contains it.
	 */
	public void setPersistentIndex(
			int index);

	/**
	 * <p>Create a cloned copy of this media entity. This is required for all targets
	 * of a strong reference to ensure that the target of a reference is owned by its
	 * source object.</p>
	 *
	 * @return Cloned copy of this media entity.
	 */
	public MediaEntity clone();

}
