/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Log: DateStruct.java,v $
 * Revision 1.7  2011/02/14 22:32:49  vizigoth
 * First commit after major sourceforge outage.
 *
 * Revision 1.6  2011/01/05 13:09:06  vizigoth
 * Created new forge for making record and union type values.
 *
 * Revision 1.5  2011/01/04 10:40:23  vizigoth
 * Refactor all package names to simpler forms more consistent with typical Java usage.
 *
 * Revision 1.4  2009/05/14 16:15:24  vizigoth
 * Major refactor to remove dependency on JPA and introduce better interface and implementation separation. Removed all setPropertiesFromInterface and castFromInterface methods.
 *
 * Revision 1.3  2008/01/14 20:52:23  vizigoth
 * Changed terminology for interfaces so that they all specify something.
 *
 * Revision 1.2  2007/12/14 15:01:49  vizigoth
 * Added to and edited document comments to a release level.
 *
 * Revision 1.1  2007/11/13 22:13:11  vizigoth
 * Public release of MAJ API.
 */

package tv.amwa.maj.record;

import java.util.Calendar;

import tv.amwa.maj.integer.Int16;
import tv.amwa.maj.integer.UInt8;

/** 
 * <p>Specifies the date component of {@linkplain TimeStamp timestamp values} 
 * that are specified according to Coordinated Universal Time (UTC). AAF represents dates with year,
 * month and day of the month components.</p> 
 * 
 * <p>To make values of this type, use the following methods from the 
 * {@linkplain tv.amwa.maj.industry.Forge MAJ forge}:</p>
 * 
 * <ul>
 *  <li>From its component values: {@link tv.amwa.maj.industry.Forge#makeDate(byte, byte, short)};</li>
 *  <li>From an existing Java Calendar value: {@link tv.amwa.maj.industry.Forge#makeDate(Calendar)};</li>
 *  <li>For today's date: {@link tv.amwa.maj.industry.Forge#todaysDate()};</li>
 *  <li>From a string representation similar to an XML schema date type as generated by {@link #toString()}: 
 *  {@link tv.amwa.maj.industry.Forge#parseDate(String)}.</li>
 * </ul>
 * 
 * @see tv.amwa.maj.industry.TypeDefinitions#DateStruct
 * @see TimeStamp
 * @see TimeStruct
 * 
 *
*/

public interface DateStruct { 
	
	/**
	 * <p>Simultaneously set the components of date structure value with the given day, 
	 * month and year.</p>
	 * 
	 * @param day Day of the month component of the date.
	 * @param month Month component of the date.
	 * @param year Year component of the date.
	 * 
	 * @throws IllegalArgumentException One or more of the component values is outside the acceptable 
	 * range for their values, or the specified date does not exist.
	 * 
	 */
	public void setDate(
			@UInt8 byte day, 
			@UInt8 byte month, 
			@Int16 short year)
		throws IllegalArgumentException;
	
	/** 
	 * <p>Returns the day of the month component of the date.</p>
	 * 
	 * @return Day of the month component of the date.
	 */
	public @UInt8 byte getDay();

	/**
	 * <p>Set the day of the month component of the date, in the range 1&nbsp;to&nbsp;31.</p>
	 * 
	 * @param day Day of the month component of the date.
	 * 
	 * @throws IllegalArgumentException Provided day value is outside the acceptable range.
	 * 
	 */
	public void setDay(
			@UInt8 byte day) 
		throws IllegalArgumentException;

	/** 
	 * <p>Returns the month component of the date, which is in the range 1&nbsp;to&nbsp;12.</p>
	 * 
	 * @return Month component of the date.
	 */
	public @UInt8 byte getMonth();

	/** 
	 * <p>Set the month component of the date, which is in the range 1&nbsp;to&nbsp;12.</p>
	 * 
	 * @param month Month component of the date.
	 * 
	 * @throws IllegalArgumentException Provided month value is outside the acceptable range.
	 */
	public void setMonth(
			@UInt8 byte month)
		throws IllegalArgumentException;

	/** 
	 * <p>Returns the year component of the date.</p>
	 * 
	 * @return Year part of the date.
	 */
	public @Int16 short getYear();

	/** 
	 * <p>Sets the year component of the date, in the range -9999&nbsp;to&nbsp;9999.</p>
	 * 
	 * @param year Year component of the date.
	 * 
	 * @throws IllegalArgumentException Year is outside the acceptable range.
	 */
	public void setYear(
			@Int16 short year)
		throws IllegalArgumentException;

	/**
	 * <p>Returns a {@link java.util.Calendar} value representing the 
	 * date represented by this value.</p>
	 * 
	 * @return Java Calendar representation of this date structure, to the current default 
	 * timezone and locale settings.
	 */
	public Calendar getDateValue();
	
	/**
	 * <p>Create a cloned copy of this date value.</p>
	 * 
	 * @return Cloned copy of this date value.
	 */
	public DateStruct clone();
	
	/**
	 * <p>Formats the date value as a UTC date-only value. The format is 
	 * "<code>yyyy-mm-dd</code>", the year followed by the month followed by
	 * the day of the month. Note that locale and timezone are not available
	 * in a date structure.</p>
	 * 
	 * @return Date value formatted as a UTC-style string without a timezone.
	 * 
	 * @see tv.amwa.maj.industry.Forge#parseDate(String)
	 */
	public String toString();
}
