/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Log: InstanceNumberGeneration.java,v $
 * Revision 1.3  2011/07/27 17:23:19  vizigoth
 * Removed import dependency on classes of the implementation package.
 *
 * Revision 1.2  2011/02/14 22:32:49  vizigoth
 * First commit after major sourceforge outage.
 *
 * Revision 1.1  2011/01/05 13:09:06  vizigoth
 * Created new forge for making record and union type values.
 *
 * Revision 1.1  2011/01/04 10:42:32  vizigoth
 * Refactor all package names to simpler forms more consistent with typical Java usage.
 *
 * Revision 1.5  2009/05/14 16:15:30  vizigoth
 * Major refactor to remove dependency on JPA and introduce better interface and implementation separation. Removed all setPropertiesFromInterface and castFromInterface methods.
 *
 * Revision 1.4  2009/03/30 09:05:03  vizigoth
 * Refactor to use SMPTE harmonized names and add early KLV file support.
 *
 * Revision 1.3  2008/10/15 09:45:54  vizigoth
 * Edited documentation to release standard.
 *
 * Revision 1.2  2008/03/07 08:08:09  vizigoth
 * Edited comments to release standard.
 *
 * Revision 1.1  2007/11/13 22:14:43  vizigoth
 * Public release of MAJ API.
 */

package tv.amwa.maj.record;


/** 
 * <p>Describes different methods used to generate the instance number part of a UMID, which forms bytes&nbsp;14,
 * 15 and&nbsp;16 of a UMID value. The techniques represented by this enumeration are defined in appendix&nbsp;B 
 * of SMPTE&nbsp;330M-2004. The next instance of a UMID can be generated by calling {@link PackageID#nextInstance()}.</p>
 * 
 * <p>To find the corresponding byte value of the instance number generation type, call
 * {@link #getMethodCode()}. The static method {@link #generationMethodFromCode(byte)} converts the
 * byte value code into a value of this enumeration.</p>
 * 
 * <p>The instance number generation method is encoded within a UMID value. Call {@link PackageID#getInstanceGenerator()}
 * to find out the kind of instance number generation in use.</p>
 *
 * @see tv.amwa.maj.industry.Forge#generatePackageID(tv.amwa.maj.enumeration.MaterialType, InstanceNumberGeneration, MaterialNumberGeneration)
 * @see tv.amwa.maj.industry.Forge#generatePackageID(tv.amwa.maj.enumeration.MaterialType, InstanceNumberGeneration, MaterialNumberGeneration, AUID)
 * @see tv.amwa.maj.industry.Forge#generatePackageID(tv.amwa.maj.enumeration.MaterialType, InstanceNumberGeneration, MaterialNumberGeneration, byte[])
 * @see PackageID#nextInstance()
 *
 *
 *
 */
public enum InstanceNumberGeneration {

	/** 
	 * <p>No defined method for instance generation.</p>
	 */
	NotDefined((byte) 0x00),
	/** 
	 * <p>Local registration where the new instance number count is incremented by one for each new instance.</p>
	 */
	LocalRegistration((byte) 0x01),
	/** 
	 * <p>Instance number is generated using a 24-bit pseudo random number generator.</p>
	 */
	PseudoRandom24Bit((byte) 0x02),
	/** 
	 * <p>Instance number is generated by incrementing an 8-bit counter alongside a 16-bit pseudo random number 
	 * generator.</p>
	 */
	CopyAndPseudoRandom16Bit((byte) 0x03),
	/** 
	 * <p>Instance number is generated from data from an associated live data stream.</p> 
	 */
	LiveStream((byte) 0x0F);
	
	/** Internal representation of the method code. */
	private byte method;
	
	/**
	 * <p>Create an instance of this enumeration as and when required.</p>
	 *
	 * @param method Byte code value representing the instance number generation type.
	 */
	private InstanceNumberGeneration(byte method) {
		
		this.method = method;
	}

	/**
	 * <p>Returns the numeric code for the UMID instance number generation type, as defined in table&nbsp;4
	 * of SMPTE&nbsp;330M.</p>
	 *
	 * @return Numeric code of the instance number generator.
	 */
	public byte getMethodCode() {
		
		return method;
	}

	/**
	 * <p>Converts a numeric code for a instance number generation type into a value of this enumeration, 
	 * as defined according to table&nbsp;4 of SMPTE&nbsp;330M.</p>
	 *
	 * @param code Numeric ode to convert to an enumeration value.
	 * @return Enumeration value corresponding to the code, or <code>null</code> if no corresponding
	 * enumeration value could be found.
	 */
	public final static InstanceNumberGeneration generationMethodFromCode(
			byte code) {
		
		for ( InstanceNumberGeneration generation : InstanceNumberGeneration.values() )
			if (generation.getMethodCode() == code) return generation;
		
		return null;
	}
}
