/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Log: ProductVersion.java,v $
 * Revision 1.7  2011/01/05 13:09:06  vizigoth
 * Created new forge for making record and union type values.
 *
 * Revision 1.6  2011/01/04 10:40:23  vizigoth
 * Refactor all package names to simpler forms more consistent with typical Java usage.
 *
 * Revision 1.5  2009/05/14 16:15:24  vizigoth
 * Major refactor to remove dependency on JPA and introduce better interface and implementation separation. Removed all setPropertiesFromInterface and castFromInterface methods.
 *
 * Revision 1.4  2009/03/30 09:05:00  vizigoth
 * Refactor to use SMPTE harmonized names and add early KLV file support.
 *
 * Revision 1.3  2008/10/16 16:52:01  vizigoth
 * First early release 0.1.
 *
 * Revision 1.2  2007/12/14 15:01:49  vizigoth
 * Added to and edited document comments to a release level.
 *
 * Revision 1.1  2007/11/13 22:13:05  vizigoth
 * Public release of MAJ API.
 */

package tv.amwa.maj.record;

import tv.amwa.maj.enumeration.ProductReleaseType;
import tv.amwa.maj.integer.UInt16;

/** <p>Specifies the version number of an application, represented with four release levels
 * and its {@linkplain tv.amwa.maj.enumeration.ProductReleaseType product release type}. 
 * The four release levels specify the major, minor, tertiary and patch level of a product.</p>
 * 
 * <p>To make values of this type, use the following methods from the 
 * {@linkplain tv.amwa.maj.industry.Forge MAJ forge}:</p>
 * 
 * <ul>
 *  <li>From its constituent parts: 
 *  {@link tv.amwa.maj.industry.Forge#makeProductVersion(short, short, short, short, ProductReleaseType)};</li>
 *  <li>A zero <em>default</em> value: {@link tv.amwa.maj.industry.Forge#zeroProductVersion()};
 *  <li>From a pseudo-XML string representation as generated by {@link #toString()}: 
 *  {@link tv.amwa.maj.industry.Forge#parseProductVersion(String)}.</li>
 * </ul> 
 * 
 * @see tv.amwa.maj.industry.TypeDefinitions#ProductVersionType
 * @see VersionType
 * 
 *
*/

public interface ProductVersion { 

	/**
	 * <p>Returns the major component of the version number.</p>
	 *
	 * @return Major component of the version number.
	 */
	public @UInt16 short getMajor();

	/**
	 * <p>Sets the major component of this version number.</p>
	 *
	 * @param major Major component of the version number.
	 * 
	 * @throws IllegalArgumentException Cannot set the major component of this product version
	 * to a negative value.
	 */
	public void setMajor(
			@UInt16 short major)
		throws IllegalArgumentException;

	/**
	 * <p>Returns the minor component of the version number.</p>
	 *
	 * @return Minor component of the version number.
	 */
	public @UInt16 short getMinor();

	/**
	 * <p>Sets the minor component of the version number.</p>
	 *
	 * @param minor Minor component of the version number.
	 * 
	 * @throws IllegalArgumentException Cannot set the minor component of this product version
	 * to a negative value.
	 */
	public void setMinor(
			@UInt16 short minor)
		throws IllegalArgumentException;

	/**
	 * <p>Returns the tertiary component of the version number.</p>
	 *
	 * @return Tertiary component of the version number.
	 */
	public @UInt16 short getTertiary();

	/**
	 * <p>Sets the tertiary component of the version number.</p>
	 *
	 * @param tertiary Tertiary component of the version number.
	 * 
	 * @throws IllegalArgumentException Cannot set the tertiary component of this product version
	 * to a negative value.
	 */
	public void setTertiary(
			@UInt16 short tertiary)
		throws IllegalArgumentException;

	/**
	 * <p>Returns the patch level component of the version number.</p>
	 *
	 * @return Patch level component of the version number.
	 */
	public @UInt16 short getPatchLevel();

	/**
	 * <p>Sets the patch level component of the version number.</p>
	 *
	 * @param patchLevel Patch level component of the version number.
	 * 
	 * @throws IllegalArgumentException Cannot set the patch level component of this product version
	 * to a negative value.
	 */
	public void setPatchLevel(
			@UInt16 short patchLevel)
		throws IllegalArgumentException;
	
	/**
	 * <p>Returns the associated application's product release type, for example final, debug, beta etc..</p>
	 *
	 * @return Associated application's release type.
	 */
	public ProductReleaseType getBuildType();

	/**
	 * <p>Sets the associated application's product release type, for example final, debug, beta etc..</p>
	 *
	 * @param type Associated application's release type.
	 * 
	 * @throws NullPointerException The given product release type is <code>null</code>.
	 */
	public void setBuildType(
			ProductReleaseType type) 
		throws NullPointerException;

	/**
	 * <p>Create a cloned copy of this product version.</p>
	 * 
	 * @return Cloned copy of this product version.
	 */
	public ProductVersion clone();
	
	/**
	 * <p>Formats a pseudo-XML representation of the current value of this product version. 
	 * The value will be formatted in a similar way to the following example:</p>
	 * 
	 * <pre>
	 *     &lt;ProductVersion&gt;
	 *       &lt;Major&gt;0&lt;/Major&gt;
	 *       &lt;Minor&gt;3&lt;/Minor&gt;
	 *       &lt;Tertiary&gt;1&lt;/Tertiary&gt;
	 *       &lt;PatchLevel&gt;2&lt;/PatchLevel&gt;
	 *       &lt;BuildType&gt;VersionDebug&lt;/BuildType&gt;
	 *     &lt;/ProductVersion&gt;
	 * </pre>
	 * 
	 * @return Pseudo-XML representation of this product version. 
	 */
	public String toString();
}
