/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Log: AUID.java,v $
 * Revision 1.11  2011/02/14 22:32:49  vizigoth
 * First commit after major sourceforge outage.
 *
 * Revision 1.10  2011/01/05 13:09:06  vizigoth
 * Created new forge for making record and union type values.
 *
 * Revision 1.9  2011/01/04 10:40:23  vizigoth
 * Refactor all package names to simpler forms more consistent with typical Java usage.
 *
 * Revision 1.8  2010/03/19 10:15:09  vizigoth
 * Pulled isUniversalLabel method up into the interface.
 *
 * Revision 1.7  2009/05/14 16:15:24  vizigoth
 * Major refactor to remove dependency on JPA and introduce better interface and implementation separation. Removed all setPropertiesFromInterface and castFromInterface methods.
 *
 * Revision 1.6  2009/03/30 09:05:00  vizigoth
 * Refactor to use SMPTE harmonized names and add early KLV file support.
 *
 * Revision 1.5  2008/10/16 16:52:01  vizigoth
 * First early release 0.1.
 *
 * Revision 1.4  2008/02/08 12:43:04  vizigoth
 * Consistent referal to zero rather than null/nil mob id and isOriginal/Contextual methods added to the interface.
 *
 * Revision 1.3  2008/01/14 20:52:23  vizigoth
 * Changed terminology for interfaces so that they all specify something.
 *
 * Revision 1.2  2007/12/14 15:01:49  vizigoth
 * Added to and edited document comments to a release level.
 *
 * Revision 1.1  2007/11/13 22:12:49  vizigoth
 * Public release of MAJ API.
 */

package tv.amwa.maj.record;

import java.io.Serializable;

import tv.amwa.maj.integer.UInt16;
import tv.amwa.maj.integer.UInt32;
import tv.amwa.maj.integer.UInt8Array;


/** 
 * <p>Specifies a 16-byte unique identifier whose value is a 
 * SMPTE&nbsp;298M Universal Label or a UUID or GUID.</p>
 * 
 * <p>UUIDs are defined in <a href="http://www.faqs.org/rfcs/rfc4122.html" alt="RFC 4122">rfc&nbsp;4122</a>. 
 * For media metadata, SMPTE have defined a dictionary for Universal Labels, which are representable by
 * AUID values, that is defined according to SMPTE&nbsp;335M "Metadata Dictionary Structure". 
 * This current version of the dictionary can be accessed via the 
 * <a href="http://www.smpte-ra.org/">SMPTE Registration Authority's website</a>.</p>
 * 
 * <p>A Universal label is represented as an AUID by storing bytes 9&nbsp;to&nbsp;16 of the label
 * in bytes 1&nbsp;to&nbsp;8 of an AUID and bytes 1&nbsp;to&nbsp;8 of the same Universal label in bytes
 * 9&nbsp;to&nbsp;16 of the same AUID.</p>
 * 
 * <p>The terminology used to describe the methods of this interface is taken from the 
 * <a href="http://www.faqs.org/rfcs/rfc4122.html" alt="RFC 4122">UUID specification</a>.</p>
 * 
 * <p>To make values of this type, use the following methods from the 
 * {@linkplain tv.amwa.maj.industry.Forge MAJ forge}:</p>
 * 
 * <ul>
 *  <li>From its underlying bytes: {@link tv.amwa.maj.industry.Forge#makeAUID(byte[])};</li>
 *  <li>From its constituent parts: {@link tv.amwa.maj.industry.Forge#makeAUID(int, short, short, byte[])};</li>
 *  <li>A time-based UUID: {@link tv.amwa.maj.industry.Forge#timebasedAUID()} and 
 *  {@link tv.amwa.maj.industry.Forge#timebasedAUID(byte[])};</li>
 *  <li>A name-based UUID: {@link tv.amwa.maj.industry.Forge#namebasedAUID(byte[])} and 
 *  {@link tv.amwa.maj.industry.Forge#namebasedAUID(String)};</li>
 *  <li>A random UUID: {@link tv.amwa.maj.industry.Forge#randomAUID()};</li>
 *  <li>From a URN string representation as generated by {@link #toString()}: 
 *  {@link tv.amwa.maj.industry.Forge#parseAUID(String)}.</li>
 * </ul>
 * 
 * @see tv.amwa.maj.industry.TypeDefinitions#AUID
 * @see tv.amwa.maj.industry.TypeDefinitions#AUIDArray
 * @see tv.amwa.maj.industry.TypeDefinitions#AUIDSet
 * @see PackageID
 * 
 *
 */

public interface AUID 
	extends 
		Comparable<AUID>,
		Serializable { 

	/**
	 * <p>Returns the time-low part of the AUID.</p>
	 * 
	 * @return Time low part of the AUID.
	 */
	public @UInt32 int getData1();

	/**
	 * <p>Sets the time-low part of the AUID.</p>
	 * 
	 * @param data1 Time-low part of the AUID.
	 * 
	 */
	public void setData1(
			@UInt32 int data1);
	
	/**
	 * <p>Returns the time-mid part of the AUID.</p>
	 * 
	 * @return Time-mid part of the AUID.
	 */
	public @UInt16 short getData2();

	/**
	 * <p>Sets the time-mid part of the AUID.</p>
	 * 
	 * @param data2 Time-mid part of the AUID.
	 */
	public void setData2(
			@UInt16 short data2);

	/**
	 * <p>Returns the time-hi and version parts of the AUID.  This is the high part of 
	 * the timestamp value of a UUID multiplexed with the version number, where the 
	 * version number refers to what specified kind of UUID this is.</p>
	 * 
	 * @return Time-hi and version part of the AUID.
	 */
	public @UInt16 short getData3();

	/**
	 * <p>Sets the time-hi and version parts of the AUID.  This is the high part of 
	 * the timestamp value of a UUID multiplexed with the version number, where the 
	 * version number refers to the kind of UUID this is.</p>
	 * 
	 * @param data3 Time-hi and version part of the AUID.
	 */
	public void setData3(
			@UInt16 short data3);
	
	/**
	 * <p>Returns the variant, clock sequence and node parts of the AUID as an 8-byte
	 * array.</p>
	 *
	 * @return Variant, clock sequence and node parts of the AUID.
	 */
	public @UInt8Array byte[] getData4();

	/**
	 * <p>Sets the variant, clock sequence and node parts of the AUID using an 8-byte 
	 * array.</p>
	 * 
	 * @param data4 Variant, clock sequence and node parts of the AUID.
	 */
	public void setData4(
			@UInt8Array byte[] data4);

	/**
	 * <p>Tests to see if the AUID is a nil value, which means that the value of each
	 * of its components is 0. As a UUID URN, this looks like:</p>
	 * 
	 * <p><center>
	 * <code>urn:uuid:00000000-0000-0000-0000-000000000000</code>
	 * </center></p>
	 * 
	 * @return Is the AUID a nil value?
	 * 
	 * @see PackageID#isZero()
	 */
	public boolean isNil();
	
	/**
	 * <p>Create a cloned copy of this AUID value.</p>
	 * 
	 * @return Cloned copy of this AUID value.
	 */
	public AUID clone();
	
	/**
	 * <p>Returns a 16-byte array representation of this AUID value.</p>
	 * 
	 * @return Array representation of this AUID value.
	 */
	public byte[] getAUIDValue();
	
	/**
	 * <p>Determines whether this AUID value represents a SMPTE universal label.</p>
	 * 
	 * @return Does this AUID value represent a universal label?
	 * 
	 * @see tv.amwa.maj.io.mxf.UL
	 */
	public boolean isUniversalLabel();
	
	/**
	 * <p>Generates a string representation of this AUID value as a URN as specified in 
	 * section&nbsp;3 of
     * <a href="http://www.faqs.org/rfcs/rfc4122.html" alt="RFC 4122">rfc 4122</a> or
     * the commonly used URN representation of Universal Labels (starting with 
     * "<code>urn:x-ul:</code>" or "<code>urn:smpte:ul:</code>").</p>
     * 
     * @return URN representation of this AUID value.
     * 
     * @see tv.amwa.maj.industry.Forge#parseAUID(java.lang.String)
	 */
	public String toString();
}
