/*
 * Copyright 2016 Richard Cartwright
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Log: RGBAComponent.java,v $
 * Revision 1.10  2011/01/05 13:09:06  vizigoth
 * Created new forge for making record and union type values.
 *
 * Revision 1.9  2011/01/04 10:40:23  vizigoth
 * Refactor all package names to simpler forms more consistent with typical Java usage.
 *
 * Revision 1.8  2010/01/19 14:34:02  vizigoth
 * Made it cloneable.
 *
 * Revision 1.7  2009/05/14 16:15:24  vizigoth
 * Major refactor to remove dependency on JPA and introduce better interface and implementation separation. Removed all setPropertiesFromInterface and castFromInterface methods.
 *
 * Revision 1.6  2009/03/30 09:05:00  vizigoth
 * Refactor to use SMPTE harmonized names and add early KLV file support.
 *
 * Revision 1.5  2008/02/08 12:43:56  vizigoth
 * Minor comment improvement.
 *
 * Revision 1.4  2008/01/14 20:52:23  vizigoth
 * Changed terminology for interfaces so that they all specify something.
 *
 * Revision 1.3  2008/01/10 17:18:00  vizigoth
 * Minor comment link changes due to refactoring.
 *
 * Revision 1.2  2007/12/14 15:01:49  vizigoth
 * Added to and edited document comments to a release level.
 *
 * Revision 1.1  2007/11/13 22:13:08  vizigoth
 * Public release of MAJ API.
 */

package tv.amwa.maj.record;

import tv.amwa.maj.enumeration.RGBAComponentKind;
import tv.amwa.maj.integer.UInt8;

/** 
 * <p>Specifies an element of an array representing the order and size of the component 
 * values within a pixel value as part of an 
 * {@linkplain tv.amwa.maj.industry.TypeDefinitions#RGBALayout RGBA Layout}. 
 * The RGB layout type type is a fixed-size 8&nbsp;element array, where each element consists of an 
 * RGBA component value. Each RGBA component has with the following fields:</p>
 * 
 * <ul>
 * <li><code>code</code> - {@linkplain tv.amwa.maj.enumeration.RGBAComponentKind RGBA component kind enumerated value} 
 * specifying the component kind.</li>
 * <li><code>size</code> - Java byte specifying the number of bits.</li>
 * </ul>
 * 
 * <p>A {@link RGBAComponentKind#Fill fill} component indicates unused bits. After the 
 * components have been specified, the remaining <code>code</code> and <code>size</code> fields 
 * should be set to {@link RGBAComponentKind#None} and <code>0</code> respectively.</p>
 * 
 * <p>To make values of this type, use the following methods from the 
 * {@linkplain tv.amwa.maj.industry.Forge MAJ forge}:</p>
 * 
 * <ul>
 *  <li>From its code and size: 
 *  {@link tv.amwa.maj.industry.Forge#makeRGBAComponent(RGBAComponentKind, byte)};</li>
 *  <li>A zero <em>default</em> value: {@link tv.amwa.maj.industry.Forge#zeroRGBAComponent()};
 *  <li>From a pseudo-XML string representation as generated by {@link #toString()}: 
 *  {@link tv.amwa.maj.industry.Forge#parseRGBAComponent(String)}.</li>
 * </ul> 
 * 
 * @see tv.amwa.maj.industry.TypeDefinitions#RGBAComponent
 * @see tv.amwa.maj.industry.TypeDefinitions#RGBALayout
 * @see tv.amwa.maj.model.RGBADescriptor
 * 
 *
 */

public interface RGBAComponent 
	extends Cloneable { 

	/**
	 * <p>Returns the type of an RGBA component.</p>
	 *
	 * @return Type of an RGBA component.
	 */
	public RGBAComponentKind getCode();

	/**
	 * <p>Sets the type of an RGBA component.</p>
	 *
	 * @param code Type of an RGBA component.
	 * 
	 * @throws NullPointerException The given component kind is <code>null</code>.
	 */
	public void setCode(
			RGBAComponentKind code) 
		throws NullPointerException;

	/**
	 * <p>Returns the size, measured in bits, of the RGBA component.</p>
	 *
	 * @return Size of the RGBA component.
	 */
	public @UInt8 byte getComponentSize();
	
	/**
	 * <p>Sets the size, measured in bits, of the RGBA component.</p>
	 *
	 * @param size Size of the RGBA component.
	 * 
	 * @throws IllegalArgumentException The size value is negative.
	 */
	public void setComponentSize(
			@UInt8 byte size)
		throws IllegalArgumentException;
	
	/**
	 * <p>Create a cloned copy of this RGBA component value.</p>
	 * 
	 * @return Cloned copy of this RGBA component value.
	 */
	public RGBAComponent clone();
	
	/**
	 * <p>Formats a pseudo-XML string representation of this RGBA component value. The representation 
	 * consists of a <code>RGBAComponent</code> element with <code>Code</code> and <code>Size</code>
	 * sub-elements. For example, an 8-bit red component is represented as follows:</p>
	 * 
	 * <pre>
	 *   &lt;RGBAComponent&gt;
	 *     &lt;Code&gt;CompRed&lt;/Code&gt;
	 *     &lt;ComponentSize&gt;8&lt;/ComponentSize&gt;
	 *   &lt;/RGBAComponent&gt;
	 * </pre>
	 * 
	 * <p>This representation can be parsed by 
	 * {@link tv.amwa.maj.industry.Forge#parseRGBAComponent(String)}.</p>
	 * 
	 * @return Pseudo-XML representation of this RGBA component value.
	 * 
	 * @see tv.amwa.maj.industry.Forge#parseRGBAComponent(String)
	 */
	public String toString();
}
